#include "typed_log.h"

#include <yplatform/time_traits.h>

namespace xeno {

#define TYPED YLOG((*logger_), info)

namespace tl = yplatform::log::typed;

auto make_context_attr(context_ptr ctx)
{
    return tl::make_attr("context", ctx ? ctx->uniq_id() : "");
}

auto make_message_attrs(const mb::message_opt& msg)
{
    tl::attributes_map res;
    if (msg)
    {
        res << tl::make_attr("ext_size", msg->size)
            << tl::make_attr("ext_date", yplatform::time_traits::to_simple_string(msg->date))
            << tl::make_attr("mid", msg->mid);
    }
    return res;
}

auto make_phase_durations_attrs(iteration_stat_ptr stat)
{
    tl::attributes_map res;
    for (auto&& [phase, duration] : stat->phase_durations)
    {
        res << tl::make_attr(to_string(phase), time_traits::to_string(duration));
    }
    return res;
}

typed_log::typed_log(boost::asio::io_service& io, const yplatform::ptree& conf)
{
    auto typed_log_id = conf.get("typed_log_id", "tskv");
    logger_ = yplatform::log::tskv_logger(io, typed_log_id);
}

void typed_log::create_folder(
    context_ptr ctx,
    uid_t uid,
    const string& full_path,
    const mb::fid_t& parent_fid,
    error err)
{
    TYPED << make_context_attr(ctx) << tl::make_attr("type", "create_folder")
          << tl::make_attr("uid", uid) << tl::make_attr("status", err ? "error" : "ok")
          << tl::make_attr("reason", err.message()) << tl::make_attr("full_path", full_path)
          << tl::make_attr("parent_fid", parent_fid);
}

void typed_log::sync_message(
    context_ptr ctx,
    uid_t uid,
    const mb::path_t& path,
    const mb::fid_t_opt& fid,
    mb::imap_id_t ext_imap_id,
    mb::imap_id_t int_imap_id,
    const mb::message_opt& msg,
    const std::string& operation,
    error err)
{
    TYPED << make_context_attr(ctx) << tl::make_attr("type", "sync_message")
          << tl::make_attr("status", err ? "error" : "ok") << tl::make_attr("reason", err.message())
          << tl::make_attr("uid", uid) << tl::make_attr("path", path.to_string())
          << tl::make_attr("fid", fid ? *fid : "") << tl::make_attr("ext_imap_id", ext_imap_id)
          << tl::make_attr("int_imap_id", int_imap_id) << make_message_attrs(msg)
          << tl::make_attr("operation", operation);
}

void typed_log::iteration_result(context_ptr ctx, uid_t uid, iteration_stat_ptr stat, error err)
{
    TYPED << make_context_attr(ctx) << tl::make_attr("type", "iteration_result")
          << tl::make_attr("uid", uid) << tl::make_attr("status", err ? "error" : "ok")
          << tl::make_attr("reason", err.message())
          << tl::make_attr("store_message_attempts", stat->store_message_attempts)
          << tl::make_attr("store_message_errors", stat->store_message_errors)
          << tl::make_attr("downloaded_range_updated", stat->downloaded_range_updated)
          << make_phase_durations_attrs(stat);
}

}

#include <yplatform/module_registration.h>
DEFINE_SERVICE_OBJECT(xeno::typed_log)
