#pragma once

#include "get_user_addr.h"

#include <auth/authorize.h>
#include <auth/mailish_id.h>
#include <auth/get_or_create_mailish.h>
#include <common/errors.h>
#include <common/karma.h>
#include <common/http.h>
#include <common/mail_errors.h>
#include <common/types.h>
#include <mdb/accounts_repository.h>
#include <web/impl.h>
#include <web/web_context.h>
#include <web/auth/get_user_addr.h>
#include <xeno/xeno.h>

#include <ymod_mdb_sharder/shards_distributor.h>
#include <ymod_mdb_sharder/errors.h>
#include <yplatform/util/sstream.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace xeno::web {

struct auth_result
{
    uid_t uid = 0;
    std::string x_token = "";
    bool is_new_account = false;
};

using auth_cb = std::function<void(error, const auth_result&)>;

struct authorize_user_op
{
    using yield_ctx = yplatform::yield_context<authorize_user_op>;
    using stream_ptr = ymod_webserver::http::stream_ptr;

    authorize_user_op(
        web_context_ptr web_ctx,
        stream_ptr stream,
        ext_mailbox_ptr mailbox,
        const account_t& account,
        const string& client_id,
        const string& client_secret,
        const auth_cb& cb)
        : web_ctx(web_ctx)
        , stream(stream)
        , ext_mailbox(mailbox)
        , account(account)
        , client_id(client_id)
        , client_secret(client_secret)
        , cb(cb)
    {
        imap_retries = web_ctx->check_settings()->imap_retries;
        macs = yplatform::find<ymod_macs::module>("macs");
        shards_distributor = yplatform::find<ymod_mdb_sharder::shards_distributor, std::shared_ptr>(
            "shards_distributor");
        web = yplatform::find<web::impl, std::shared_ptr>("xeno_web");
    }

    void operator()(yield_ctx ctx, error ec = {})
    {
        try
        {
            reenter(ctx)
            {
                if (account.auth_data.size() != 1)
                {
                    throw std::runtime_error(
                        "wrong authorize_user_op call, account should have exactly 1 auth_data");
                }

                do
                {
                    yield ext_mailbox->imap_authorize(
                        account.imap_login, *account.auth_data.begin(), account.imap_ep, ctx);
                    WEB_LOG_STREAM(stream, info)
                        << "authorize_user_op: imap auth result: " << ec.message();
                } while (ec && ec != errc::external_auth_invalid && imap_retries-- > 0);

                if (ec) yield break;

                if (smtp_params_incomplete())
                {
                    ec = web_errors::incomplete_params;
                    yield break;
                }

                yield ext_mailbox->check_smtp_credentials(
                    account.smtp_login,
                    *account.auth_data.begin(),
                    account.smtp_ep,
                    account.email,
                    ctx);
                if (ec) yield break;
                WEB_LOG_STREAM(stream, info) << "authorize_user_op: success smtp validation";

                mailish_id = make_mailish_id(account);

                WEB_LOG_STREAM(stream, info)
                    << "authorize_user_op: getting mailish account from passport";
                yield auth::get_or_create_mailish(
                    stream->ctx(),
                    account.email,
                    mailish_id,
                    client_id,
                    client_secret,
                    stream->request()->url.params,
                    get_user_ip(stream),
                    ctx);
                if (ec) yield break;

                WEB_LOG_STREAM(stream, info) << "authorize_user_op: getting x-token id";
                yield auth::authorize(
                    stream->request()->ctx(),
                    result.x_token,
                    get_user_ip(stream),
                    get_user_port(stream),
                    ctx);
                if (ec) yield break;

                if (is_karma_bad(karma))
                {
                    ec = ::xeno::code::bad_karma;
                    yield break;
                }

                WEB_LOG_STREAM(stream, info) << "authorize_user_op: getting user shard";
                yield macs->get_user_shard_id(stream->ctx(), std::to_string(account.uid), ctx);
                if (ec) yield break;

                WEB_LOG_STREAM(stream, info) << "authorize_user_op: saving account to mdb";
                yield save_account(ctx);
                if (ec) yield break;

                WEB_LOG_STREAM(stream, info)
                    << "authorize_user_op: getting xeno owner node for account";
                yield shards_distributor->get_owner(stream->ctx(), shard_id, ctx);
                if (ec == ymod_mdb_sharder::error::not_owned)
                {
                    WEB_LOG_STREAM(stream, info)
                        << "authorize_user_op: owner doesn't exists, can't load new user";
                    ec = {};
                }
                else if (ec)
                {
                    yield break;
                }
                else
                {
                    if (account_owner.id == shards_distributor->my_node_id())
                    {
                        WEB_LOG_STREAM(stream, info)
                            << "authorize_user_op: loading new user on current node";
                        yplatform::find<::xeno::xeno>("xeno")->load_user(account.uid, shard_id);
                    }
                    else
                    {
                        WEB_LOG_STREAM(stream, info)
                            << "authorize_user_op: loading new user on " << account_owner.host;
                        yield load_account_on_owner_node(ctx);
                    }
                }
            }
        }
        catch (const std::exception& e)
        {
            WEB_LOG_STREAM(stream, error) << "authorize_user_op: exception " << e.what();
            ec = ::xeno::code::operation_exception;
        }
        if (ctx.is_complete())
        {
            cb(ec, result);
        }
    }

    void operator()(yield_ctx ctx, error ec, const auth::get_or_create_mailish_response& resp)
    {
        if (!ec)
        {
            inject_uid(resp.uid);
            account.uid = resp.uid;
            result.uid = resp.uid;
            result.is_new_account = resp.is_new_account;
            result.x_token = resp.xtoken;
        }
        (*this)(ctx, ec);
    }

    void operator()(yield_ctx ctx, error ec, auth::auth_response_ptr auth_resp)
    {
        if (!ec)
        {
            auto auth_data = account.auth_data.begin();
            auth_data->xtoken_id = auth_resp->xtoken_id;
            karma = auth_resp->karma;
        }
        (*this)(ctx, ec);
    }

    void operator()(yield_ctx ctx, mail_errors::error_code ec, const shard_id& shard_id)
    {
        if (!ec)
        {
            this->shard_id = shard_id;
            (*this)(ctx, {});
        }
        else
        {
            WEB_LOG_STREAM(stream, error)
                << "authorize_user_op: can't get user shard: " << mail_error_message(ec);
            (*this)(ctx, ::xeno::code::cannot_get_user_shard_id);
        }
    }

    void operator()(yield_ctx ctx, error ec, const ymod_mdb_sharder::node_info& owner)
    {
        if (!ec)
        {
            account_owner = owner;
        }
        (*this)(ctx, ec);
    }

    void operator()(yield_ctx ctx, error err, yhttp::response resp)
    {
        error ec = ::xeno::code::ok;
        if (err || resp.status != 200)
        {
            if (err)
            {
                WEB_LOG_STREAM(stream, error)
                    << "authorize_user_op: proxy update account: " << err.message();
            }
            else
            {
                WEB_LOG_STREAM(stream, error)
                    << "authorize_user_op: proxy update account bad response: " << resp.status
                    << " " << resp.reason;
            }
            // we should return ok because user has been authorized successfully
        }
        (*this)(ctx, ec);
    }

    void save_account(yield_ctx ctx)
    {
        auto accounts_repository =
            std::make_shared<mdb::accounts_repository>(stream->ctx(), account.uid, shard_id, false);
        accounts_repository->save_account(account, ctx);
    }

    void load_account_on_owner_node(yield_ctx ctx)
    {
        std::string url;
        yplatform::sstream url_stream(url);
        url_stream << account_owner.host << ":" << web_ctx->proxy_settings()->internal_api_port;
        auto client = web_ctx->get_http_client(url);
        auto req = yhttp::request::GET("/load_user?uid=" + std::to_string(account.uid));
        client->async_run(stream->ctx(), std::move(req), ctx);
    }

    bool smtp_params_incomplete()
    {
        return account.smtp_login.empty() || account.auth_data.begin()->smtp_credentials.empty() ||
            account.smtp_ep.host.empty() || !account.smtp_ep.port;
    }

    void inject_uid(const uid_t& uid)
    {
        APPEND_UID_TO_LOG_PREFIX(stream, uid);
        update_custom_log_param(stream, "uid", std::to_string(uid));
    }

    web_context_ptr web_ctx;
    stream_ptr stream;
    ext_mailbox_ptr ext_mailbox;
    account_t account;
    std::string mailish_id;
    std::string client_id;
    std::string client_secret;
    auth_cb cb;

    boost::shared_ptr<ymod_macs::module> macs;
    std::shared_ptr<ymod_mdb_sharder::shards_distributor> shards_distributor;
    std::shared_ptr<web::impl> web;
    int imap_retries;

    ::xeno::shard_id shard_id;
    ymod_mdb_sharder::node_info account_owner;
    auth_result result;
    karma_t karma;
};

}

#include <yplatform/unyield.h>
