#pragma once

#include <common/types.h>

#include <string>
#include <vector>

namespace xeno::web {

enum web_errors : int
{
    ok = 0,
    exception_occured,
    domain_not_found,
    ispdb_error,
    bad_request,
    incomplete_params,
    forbidden_provider,
    rate_limiter_error,
    rate_limit_exceeded,
    check_yandex_by_mx_error,
    node_not_found,
    folder_locked,
    auth_error,
    account_not_ready,
    proxy_error,

    COUNT
    // DO NOT FORGET TO EXTEND error_names
};

inline const std::string& error_message(web_errors err)
{
    static constexpr std::initializer_list<const char*> names = { "success",
                                                                  "exception occured",
                                                                  "domain not found",
                                                                  "ispdb error",
                                                                  "bad request",
                                                                  "incomplete params",
                                                                  "forbidden provider",
                                                                  "rate limiter error",
                                                                  "rate limit exceeded",
                                                                  "check yandex by mx error",
                                                                  "node not found",
                                                                  "folder locked",
                                                                  "auth error",
                                                                  "account not ready",
                                                                  "proxy error" };
    static_assert(
        names.size() == web_errors::COUNT,
        "Error codes count doesn't correspond with error names count");
    static const std::vector<std::string> error_names(names.begin(), names.end());

    return error_names[err];
}

enum class web_errors_condition_category
{
    ok = 0,
    perm_error,
    COUNT
};

inline const std::string& errc_message(web_errors_condition_category err)
{
    static constexpr std::initializer_list<const char*> names = {
        "success",
        "error is permanent",
    };
    static_assert(
        names.size() == static_cast<size_t>(web_errors_condition_category::COUNT),
        "Web errors conditions count doesn't correspond with condition names count");
    static const std::vector<std::string> condition_names(names.begin(), names.end());

    return condition_names[static_cast<int>(err)];
}

}

namespace yplatform {

template <>
struct is_error_enum<xeno::web::web_errors>
{
    static const bool value = true;
};

}

namespace boost::system {

template <>
struct is_error_condition_enum<xeno::web::web_errors_condition_category>
{
    static const bool value = true;
};

}

namespace xeno::web {

const boost::system::error_category& error_category(web_errors);
const boost::system::error_category& get_error_condition_category();

inline boost::system::error_condition make_error_condition(web_errors_condition_category e)
{
    return boost::system::error_condition(static_cast<int>(e), get_error_condition_category());
}

}
