#pragma once

#include <common/domain.h>
#include <common/http.h>
#include <common/context.h>
#include <common/json.h>
#include <common/errors.h>
#include <web/errors.h>
#include <ymod_httpclient/cluster_client.h>

namespace xeno::web {

using check_yandex_by_mx_cb = std::function<void(error, bool)>;

namespace {

// check existence yandex mx entry for domain
void check_yandex_by_mx(context_ptr ctx, const std::string& email, const check_yandex_by_mx_cb& cb)
{
    auto domain = get_domain_by_email(email);
    if (domain.empty())
    {
        YLOG(ctx->logger(), error) << "checking yandex by mx error: can't get domain by email";
        return cb(web_errors::check_yandex_by_mx_error, false);
    }
    auto request =
        http::request_t::GET("/domain/status" + yhttp::url_encode({ { "name", domain } }));
    auto client = yplatform::find<yhttp::cluster_client>("gendarme_client");
    client->async_run(ctx, std::move(request), [cb, ctx](error err, yhttp::response response) {
        try
        {
            if (err)
            {
                YLOG(ctx->logger(), error) << "checking yandex by mx error: " << err.message();
                return cb(err, false);
            }
            if (response.status != 200)
            {
                YLOG(ctx->logger(), error)
                    << "checking yandex by mx error: bad status: " << response.status << " "
                    << response.reason;
                return cb(web_errors::check_yandex_by_mx_error, false);
            }
            json::value result;
            json::reader reader;
            if (!reader.parse(response.body, result))
            {
                YLOG(ctx->logger(), error) << "checking yandex by mx error: bad json";
                return cb(web_errors::check_yandex_by_mx_error, false);
            }
            auto match = result["response"]["mx"]["match"].asBool();
            cb({}, match);
        }
        catch (const std::exception& e)
        {
            YLOG(ctx->logger(), error) << "checking yandex by mx error: " << e.what();
            cb(::xeno::code::operation_exception, false);
        }
    });
}

}
}
