#include "impl.h"
#include "web_context.h"
#include "resolver.h"
#include "auth/auth_check/auth_check_settings.h"
#include "binded_spawn.h"
#include "internal/cache_dump.h"
#include "internal/load_user.h"
#include "internal/invalidate_auth.h"
#include "internal/release_buckets.h"
#include "internal/add_bucket.h"
#include "internal/del_bucket.h"
#include "internal/add_shards_to_bucket.h"
#include "internal/del_shards_from_bucket.h"
#include "internal/process_passport_events.h"
#include "internal/acquired_buckets_info.h"
#include "internal/auth_master.h"
#include "internal/list_controllers.h"
#include "internal/providers_state.h"
#include "mobile/auth_protect.h"
#include "mobile/create_folder.h"
#include "mobile/set_read_flag.h"
#include "mobile/delete_items.h"
#include "mobile/move_items.h"
#include "mobile/clear_folder.h"
#include "mobile/delete_folder.h"
#include "mobile/update_folder.h"
#include "mobile/default_method.h"
#include "mobile/auth_by_password.h"
#include "mobile/auth_by_oauth.h"
#include "mobile/ping.h"
#include "mobile/proxy.h"
#include "mobile/store.h"
#include "mobile/set_spam_status.h"
#include "mobile/send.h"
#include "mobile/set_label.h"
#include "mobile/proxy_to_owner.h"
#include "mobile/provider.h"
#include "mobile/push.h"
#include "mobile/messages.h"
#include "mobile/proxy_to_auth_master.h"
#include "mobile/rate_limit.h"
#include "mobile/sync_status.h"
#include "mops/move.h"
#include "mops/remove.h"
#include "mops/mark.h"
#include "mops/create_folder.h"
#include "mops/update_folder.h"
#include "mops/delete_folder.h"
#include "mops/set_folder_symbol.h"
#include "mops/set_folders_order.h"
#include "mops/get_or_create_label.h"
#include "mops/set_label.h"
#include "mops/update_label.h"
#include "mops/delete_label.h"
#include "mops/get_or_create_label_by_symbol.h"
#include "sendbernar/save.h"
#include "sendbernar/send_mail.h"

#include <auth/social/settings.h>
#include <common/http.h>
#include <web/proxy_to_owner.h>

#include <yplatform/find.h>

#include <type_traits>

namespace xeno::web {

namespace {
bool validate_as_correct(const std::string&)
{
    return true;
}

template <typename Type>
struct predefined_value_converter
{
    predefined_value_converter(Type value) : value(value)
    {
    }

    Type operator()(const std::string&) const
    {
        return value;
    }

    Type value;
};

template <typename Type>
auto predefined_value(Type value)
{
    return predefined_value_converter<Type>(value);
}

template <typename Container>
struct array_converter
{
    Container operator()(const std::string& value) const
    {
        auto splitted = yplatform::util::split(value, ",");
        if constexpr (std::is_same_v<Container, decltype(splitted)>)
        {
            return splitted;
        }
        Container res(splitted.size());

        std::transform(splitted.begin(), splitted.end(), res.begin(), [](auto& entry) {
            return boost::lexical_cast<typename Container::value_type>(entry);
        });
        return res;
    }
};

bool is_num_array_member(char c)
{
    return ::isdigit(c) || c == ',';
};

template <typename... Args>
void bind_mobile_api(std::vector<std::string> paths, Args&&... args)
{
    paths.reserve(2 * paths.size());
    for (auto it = paths.rbegin(); it != paths.rend(); ++it)
    {
        paths.emplace_back("/api/mobile" + *it);
    }
    auto webserver = yplatform::find<ymod_webserver::server>("web_server");
    webserver->bind("", paths, std::forward<Args>(args)...);
}

}

namespace ph = std::placeholders;

impl::impl() : total_requests_(60)
{
}

void impl::init(const yplatform::ptree& config)
{
    auto reactor = yplatform::find_reactor(config.get<std::string>("reactor"));
    for (size_t i = 0; i < reactor->size(); ++i)
    {
        if ((*reactor)[i]->size() != 1)
            throw std::runtime_error("xeno_web is optimized for single-thread reactors - set "
                                     "pool_count=N and io_threads=1");
    }
    io_ = reactor->io();

    auto resolver_instance = std::make_shared<resolver>(config.get_child("auth.resolver"));
    auto social_settings =
        std::make_shared<auth::social::social_settings>(config.get_child("auth.social"));
    auto proxy_settings = std::make_shared<web::proxy_settings>(config.get_child("proxy"));
    auto check_settings = std::make_shared<auth_check::auth_check_settings>(
        config.get_child("auth.check_credentials"));
    auto call = yplatform::find<yhttp::call>("http_client");

    web_context::providers_ptr oauth_providers = std::make_shared<web_context::providers>();
    auto providers_cfg = config.equal_range("oauth_providers");
    for (auto& [name, provider] : boost::make_iterator_range(providers_cfg))
    {
        oauth_providers->insert(provider.data());
    }
    auto web_ctx = std::make_shared<web_context>(
        resolver_instance, social_settings, proxy_settings, check_settings, call, oauth_providers);

    auto shards_distributor =
        yplatform::find<ymod_mdb_sharder::shards_distributor>("shards_distributor");
    my_node_.host = boost::asio::ip::host_name();
    my_node_.id = shards_distributor->my_node_id();

    if (config.get<bool>("use_lease"))
    {
        auth_by_pass_resource_ = config.get("proxy_auth_by_pass_resource", "auth_by_pass_master");

        lease_node_ = yplatform::find<ymod_lease::node, std::shared_ptr>("lease_node");
        lease_node_->bind(
            auth_by_pass_resource_,
            io_->wrap(std::bind(&impl::on_busy, shared_from(this), ph::_1, ph::_2, ph::_3, ph::_4)),
            io_->wrap(std::bind(&impl::on_free, shared_from(this), ph::_1)));
    }
    else
    {
        master_node_ = my_node_;
    }

    auto rate_limiter_conf = config.get_child_optional("rate_limiter");
    if (rate_limiter_conf)
    {
        rate_limiter_ = std::make_shared<rate_limiter>(io_, *rate_limiter_conf, logger());
    }

    max_rps_ = config.get("max_rps", 1000);

    auto webserver = yplatform::find<ymod_webserver::server, std::shared_ptr>("web_server");
    using ymod_webserver::transformer;
    using ymod_webserver::argument;
    using ymod_webserver::optional_argument;
    using methods::auth_protect;
    using ymod_webserver::validator;
    using bucket_ids_t = std::vector<std::string>;
    using shard_ids_t = std::vector<std::string>;

    webserver->bind("", { "/ping" }, methods::ping);
    webserver->bind("internal", { "/ping" }, methods::ping);

    webserver->bind(
        "internal",
        { "/cache_dump" },
        internal::methods::cache_dump,
        transformer(argument<uid_t>("uid")));

    webserver->bind(
        "internal",
        { "/load_user" },
        binded_spawn<internal::methods::load_user>(web_ctx),
        transformer(
            argument<uid_t>("uid"),
            optional_argument("load", true, validate_as_correct, predefined_value(true))));

    webserver->bind(
        "internal",
        { "/unload_user" },
        binded_spawn<internal::methods::load_user>(web_ctx),
        transformer(
            argument<uid_t>("uid"),
            optional_argument("load", false, validate_as_correct, predefined_value(false))));

    webserver->bind(
        "internal",
        { "/invalidate_auth" },
        binded_spawn<internal::proxy_to_owner<internal::methods::invalidate_auth>>(web_ctx),
        transformer(argument<uid_t>("uid")));

    webserver->bind(
        "internal",
        { "/process_passport_events" },
        binded_spawn<internal::methods::process_passport_events>(web_ctx));

    webserver->bind(
        "internal", { "/acquired_buckets_info" }, internal::methods::get_acquired_buckets_info);

    webserver->bind(
        "internal",
        { "/release_buckets" },
        internal::methods::release_buckets_for,
        transformer(
            argument<bucket_ids_t>(
                "buckets_ids", validate_as_correct, array_converter<bucket_ids_t>()),
            optional_argument<unsigned int>("duration", 20u)));

    webserver->bind(
        "internal",
        { "/add_bucket" },
        internal::methods::add_bucket,
        transformer(
            argument<std::string>("bucket_id"),
            optional_argument<shard_ids_t>(
                "shard_ids", shard_ids_t(), validate_as_correct, array_converter<shard_ids_t>())));

    webserver->bind(
        "internal",
        { "/del_bucket" },
        internal::methods::del_bucket,
        transformer(argument<std::string>("bucket_id")));

    webserver->bind(
        "internal",
        { "/add_shards_to_bucket" },
        internal::methods::add_shards_to_bucket,
        transformer(
            argument<std::string>("bucket_id"),
            argument<shard_ids_t>(
                "shard_ids", validate_as_correct, array_converter<shard_ids_t>())));

    webserver->bind(
        "internal",
        { "/del_shards_from_bucket" },
        internal::methods::del_shards_from_bucket,
        transformer(
            argument<std::string>("bucket_id"),
            argument<shard_ids_t>(
                "shard_ids", validate_as_correct, array_converter<shard_ids_t>())));

    webserver->bind("internal", { "/auth_master" }, internal::methods::get_auth_master);

    webserver->bind("internal", { "/list_controllers" }, internal::methods::list_controllers);

    webserver->bind("internal", { "/providers_state" }, internal::methods::get_providers_state);

    bind_mobile_api(
        { "/v1/auth_by_password", "/v1/auth_by_login", "/v1/external_auth_by_login" },
        binded_spawn<methods::proxy_to_auth_master<methods::rate_limit<methods::auth_by_password>>>(
            web_ctx, methods::NEED_RESOLVE),
        transformer(
            argument<std::string>("email"),
            argument<std::string>("password"),
            argument<std::string>("client_id"),
            argument<std::string>("client_secret")));

    bind_mobile_api(
        { "/v1/auth_by_password_ex" },
        binded_spawn<methods::proxy_to_auth_master<methods::rate_limit<methods::auth_by_password>>>(
            web_ctx, methods::NOT_NEED_RESOLVE),
        transformer(
            argument<std::string>("email"),
            argument<std::string>("imap_login"),
            argument<std::string>("imap_password"),
            argument<std::string>("imap_host"),
            argument<uint16_t>("imap_port"),
            argument<std::string>("imap_ssl"),
            optional_argument<std::string>("smtp_login", ""),
            optional_argument<std::string>("smtp_password", ""),
            optional_argument<std::string>("smtp_host", ""),
            optional_argument<uint16_t>("smtp_port", 0u),
            optional_argument<std::string>("smtp_ssl", ""),
            argument<std::string>("client_id"),
            argument<std::string>("client_secret")));

    bind_mobile_api(
        { "/v1/auth_by_oauth", "/v1/external_auth_by_oauth" },
        binded_spawn<methods::auth_by_oauth>(web_ctx),
        transformer(
            argument<std::string>("social_task_id"),
            argument<std::string>("client_id"),
            argument<std::string>("client_secret")));

    bind_mobile_api(
        { "/v1/auth_by_external_token" },
        binded_spawn<methods::auth_by_oauth>(web_ctx),
        transformer(
            argument<std::string>("token"),
            argument<std::string>("provider"),
            argument<std::string>("application"),
            argument<std::string>("scope"),
            argument<std::string>("client_id"),
            argument<std::string>("client_secret")));

    bind_mobile_api(
        { "/v1/set_settings" },
        std::bind(binded_spawn<auth_protect<methods::proxy>>(web_ctx), ph::_1));

    bind_mobile_api(
        { "/v1/xlist" }, std::bind(binded_spawn<auth_protect<methods::proxy>>(web_ctx), ph::_1));

    bind_mobile_api(
        { "/v1/settings",
          "/v1/set_parameters",
          "/v1/threads_by_timestamp_range",
          "/v1/with_attachments",
          "/v1/only_new",
          "/v1/message_header",
          "/v1/message_body",
          "/v1/message_source",
          "/v1/attach",
          "/v1/ava",
          "/v1/ava2",
          "/v1/corp_ava",
          "/v1/search",
          "/v1/bytype",
          "/v1/suggest_contacts",
          "/v1/suggest_query",
          "/v1/attach_list",
          "/v1/check_skleyka",
          "/v1/retaksa",
          "/v1/reset_fresh",
          "/v1/reset_unvisited",
          "/v1/upload",
          "/v1/abook_top",
          "/v1/abook_suggest",
          "/v1/abook_all",
          "/v1/abook_card",
          "/v1/vdirect",
          "/v1/uaz",
          "/v1/typed_new_cnt" },
        std::bind(binded_spawn<auth_protect<methods::proxy>>(web_ctx), ph::_1));

    bind_mobile_api(
        { "/v2/search_suggest",
          "/v2/disk_ls",
          "/v2/disk_mkdir",
          "/v2/disk_operations_status",
          "/v2/disk_save",
          "/v2/disk_save_all",
          "/v2/reset_fresh",
          "/v2/search_contacts",
          "/v2/search_save",
          "/v2/uaz",
          "/v2/apns_queue_repeat",
          "/v2/classification",
          "/v2/get_tab_counters",
          "/v2/get_newsletters",
          "/v2/translate_message",
          "/v2/translation_langs",
          "/v2/quick_reply_suggestions",
          "/v2/get_disk_promocode",
          "/v2/ocr_text",
          "/v2/scan",
          "/v2/get_abook_contact",
          "/v2/generate_operation_id" },
        std::bind(binded_spawn<auth_protect<methods::proxy>>(web_ctx), ph::_1));

    bind_mobile_api(
        { "/v1/messages", "/v1/messages_by_timestamp_range" },
        std::bind(
            binded_spawn<auth_protect<mobile::proxy_to_owner<methods::messages>>>(web_ctx),
            ph::_1,
            ph::_2),
        transformer(optional_argument<bool>("skip_api_lock", false)));

    bind_mobile_api(
        { "/v1/unsubscribe_anonymous" }, std::bind(binded_spawn<methods::proxy>(web_ctx), ph::_1));

    bind_mobile_api(
        { "/v1/push" },
        std::bind(
            binded_spawn<auth_protect<mobile::proxy_to_owner<methods::push>>>(web_ctx),
            ph::_1,
            ph::_2),
        transformer(argument<std::string>("uuid")));

    bind_mobile_api(
        { "/v1/create_folder" },
        binded_spawn<auth_protect<mobile::proxy_to_owner<methods::create_folder>>>(web_ctx),
        transformer(argument<std::string>("name"), optional_argument<fid_t>("parent_fid", "")));

    bind_mobile_api(
        { "/v1/mark_read" },
        binded_spawn<auth_protect<mobile::proxy_to_owner<methods::set_read_flag>>>(web_ctx),
        transformer(
            optional_argument(
                "mids",
                mid_vector(),
                validator(is_num_array_member),
                array_converter<mid_vector>()),
            optional_argument(
                "tids",
                tid_vector(),
                validator(is_num_array_member),
                array_converter<tid_vector>()),
            optional_argument("set_read", true, validate_as_correct, predefined_value(true))));

    bind_mobile_api(
        { "/v1/mark_unread" },
        binded_spawn<auth_protect<mobile::proxy_to_owner<methods::set_read_flag>>>(web_ctx),
        transformer(
            optional_argument(
                "mids",
                mid_vector(),
                validator(is_num_array_member),
                array_converter<mid_vector>()),
            optional_argument(
                "tids",
                tid_vector(),
                validator(is_num_array_member),
                array_converter<tid_vector>()),
            optional_argument("set_read", false, validate_as_correct, predefined_value(false))));

    bind_mobile_api(
        { "/v1/delete_items" },
        binded_spawn<auth_protect<mobile::proxy_to_owner<methods::delete_items>>>(web_ctx),
        transformer(
            optional_argument(
                "mids",
                mid_vector(),
                validator(is_num_array_member),
                array_converter<mid_vector>()),
            optional_argument(
                "tids",
                tid_vector(),
                validator(is_num_array_member),
                array_converter<tid_vector>()),
            argument<fid_t>("current_folder")));

    bind_mobile_api(
        { "/v2/delete_items" },
        binded_spawn<auth_protect<mobile::proxy_to_owner<methods::delete_items>>>(web_ctx),
        transformer(
            optional_argument("purge", false, validate_as_correct, predefined_value(false))));

    bind_mobile_api(
        { "/v2/purge_items" },
        binded_spawn<auth_protect<mobile::proxy_to_owner<methods::delete_items>>>(web_ctx),
        transformer(optional_argument("purge", true, validate_as_correct, predefined_value(true))));

    bind_mobile_api(
        { "/v1/clear_folder" },
        binded_spawn<auth_protect<mobile::proxy_to_owner<methods::clear_folder>>>(web_ctx),
        transformer(argument<fid_t>("fid")));

    bind_mobile_api(
        { "/v1/delete_folder" },
        binded_spawn<auth_protect<mobile::proxy_to_owner<methods::delete_folder>>>(web_ctx),
        transformer(argument<fid_t>("fid")));

    bind_mobile_api(
        { "/v1/update_folder" },
        binded_spawn<auth_protect<mobile::proxy_to_owner<methods::update_folder>>>(web_ctx),
        transformer(
            argument<fid_t>("fid"),
            optional_argument("name", std::string()),
            optional_argument(
                "parent_fid",
                fid_t_opt(),
                ymod_webserver::detail::default_validator<std::string>(),
                ymod_webserver::detail::converter<std::string>())));

    bind_mobile_api(
        { "/v1/move_to_folder" },
        binded_spawn<auth_protect<mobile::proxy_to_owner<methods::move_items>>>(web_ctx),
        transformer(
            argument<fid_t>("fid"),
            optional_argument(
                "mids",
                mid_vector(),
                validator(is_num_array_member),
                array_converter<mid_vector>()),
            optional_argument(
                "tids",
                tid_vector(),
                validator(is_num_array_member),
                array_converter<tid_vector>())));

    bind_mobile_api(
        { "/v1/store" },
        binded_spawn<auth_protect<mobile::proxy_to_owner<methods::store>>>(web_ctx));
    bind_mobile_api(
        { "/v1/send" }, binded_spawn<auth_protect<mobile::proxy_to_owner<methods::send>>>(web_ctx));

    bind_mobile_api(
        { "/v1/foo" },
        binded_spawn<auth_protect<mobile::proxy_to_owner<methods::set_spam_status>>>(web_ctx),
        transformer(
            optional_argument(
                "mids",
                mid_vector(),
                validator(is_num_array_member),
                array_converter<mid_vector>()),
            optional_argument(
                "tids",
                tid_vector(),
                validator(is_num_array_member),
                array_converter<tid_vector>()),
            optional_argument<fid_t>("current_folder", ""),
            optional_argument("spam", true, validate_as_correct, predefined_value(true))));

    bind_mobile_api(
        { "/v1/antifoo" },
        binded_spawn<auth_protect<mobile::proxy_to_owner<methods::set_spam_status>>>(web_ctx),
        transformer(
            optional_argument(
                "mids",
                mid_vector(),
                validator(is_num_array_member),
                array_converter<mid_vector>()),
            optional_argument(
                "tids",
                tid_vector(),
                validator(is_num_array_member),
                array_converter<tid_vector>()),
            optional_argument<fid_t>("current_folder", ""),
            optional_argument("spam", false, validate_as_correct, predefined_value(false))));

    bind_mobile_api(
        { "/v1/mark_with_label" },
        binded_spawn<auth_protect<mobile::proxy_to_owner<methods::set_label>>>(web_ctx),
        transformer(
            optional_argument(
                "mids",
                mid_vector(),
                validator(is_num_array_member),
                array_converter<mid_vector>()),
            optional_argument(
                "tids",
                tid_vector(),
                validator(is_num_array_member),
                array_converter<tid_vector>()),
            argument<lid>("lid"),
            optional_argument<bool>("mark", false)));

    bind_mobile_api(
        { "/v1/provider" },
        binded_spawn<auth_protect<mobile::proxy_to_owner<methods::provider>>>(web_ctx));

    bind_mobile_api(
        { "/v1/sync_status" },
        binded_spawn<auth_protect<mobile::proxy_to_owner<methods::sync_status>>>(web_ctx));

    webserver->bind("", { "*rate-limit*" }, [](stream_ptr stream) {
        respond(stream, make_error_response(stream, web_errors::rate_limit_exceeded));
    });

    webserver->bind(
        "",
        { "/mops/move" },
        binded_spawn<proxy_to_owner<mops::response, mops::methods::move>>(web_ctx),
        transformer(
            argument<uid_t>("uid"),
            argument<fid_t>("fid"),
            optional_argument(
                "mids",
                mid_vector(),
                validator(is_num_array_member),
                array_converter<mid_vector>())));

    webserver->bind(
        "",
        { "/mops/remove" },
        binded_spawn<proxy_to_owner<mops::response, mops::methods::remove>>(web_ctx),
        transformer(
            argument<uid_t>("uid"),
            optional_argument(
                "mids",
                mid_vector(),
                validator(is_num_array_member),
                array_converter<mid_vector>())));

    webserver->bind(
        "",
        { "/mops/mark" },
        binded_spawn<proxy_to_owner<mops::response, mops::methods::mark>>(web_ctx),
        transformer(
            argument<uid_t>("uid"),
            argument<std::string>("status"),
            optional_argument(
                "mids",
                mid_vector(),
                validator(is_num_array_member),
                array_converter<mid_vector>())));

    webserver->bind(
        "",
        { "/mops/create_folder", "/mops/get_or_create_folder" },
        binded_spawn<proxy_to_owner<mops::response, mops::methods::create_folder>>(web_ctx),
        transformer(
            argument<uid_t>("uid"),
            argument<std::string>("name"),
            optional_argument<fid_t>("parent_fid", ""),
            optional_argument<std::string>("symbol", "")));

    webserver->bind(
        "",
        { "/mops/update_folder" },
        binded_spawn<proxy_to_owner<mops::response, mops::methods::update_folder>>(web_ctx),
        transformer(
            argument<uid_t>("uid"),
            argument<fid_t>("fid"),
            argument<std::string>("name"),
            optional_argument<fid_t>("parent_fid", "")));

    webserver->bind(
        "",
        { "/mops/delete_folder" },
        binded_spawn<proxy_to_owner<mops::response, mops::methods::delete_folder>>(web_ctx),
        transformer(argument<uid_t>("uid"), argument<fid_t>("fid")));

    webserver->bind(
        "",
        { "/mops/create_label" },
        binded_spawn<proxy_to_owner<mops::response, mops::methods::get_or_create_label>>(web_ctx),
        transformer(
            argument<uid_t>("uid"),
            argument<std::string>("name"),
            argument<std::string>("color"),
            argument<std::string>("type"),
            optional_argument("force_create", true, validate_as_correct, predefined_value(true))));

    webserver->bind(
        "",
        { "/mops/get_or_create_label" },
        binded_spawn<proxy_to_owner<mops::response, mops::methods::get_or_create_label>>(web_ctx),
        transformer(
            argument<uid_t>("uid"),
            argument<std::string>("name"),
            argument<std::string>("color"),
            argument<std::string>("type"),
            optional_argument(
                "force_create", false, validate_as_correct, predefined_value(false))));

    webserver->bind(
        "",
        { "/mops/label" },
        binded_spawn<proxy_to_owner<mops::response, mops::methods::set_label>>(web_ctx),
        transformer(
            argument<uid_t>("uid"),
            optional_argument(
                "mids",
                mid_vector(),
                validator(is_num_array_member),
                array_converter<mid_vector>()),
            optional_argument<lid_vector>(
                "lids", lid_vector(), validate_as_correct, array_converter<lid_vector>()),
            optional_argument("mark", true, validate_as_correct, predefined_value(true))));

    webserver->bind(
        "",
        { "/mops/unlabel" },
        binded_spawn<proxy_to_owner<mops::response, mops::methods::set_label>>(web_ctx),
        transformer(
            argument<uid_t>("uid"),
            optional_argument(
                "mids",
                mid_vector(),
                validator(is_num_array_member),
                array_converter<mid_vector>()),
            optional_argument<lid_vector>(
                "lids", lid_vector(), validate_as_correct, array_converter<lid_vector>()),
            optional_argument("mark", false, validate_as_correct, predefined_value(false))));

    webserver->bind(
        "",
        { "/mops/update_label" },
        binded_spawn<proxy_to_owner<mops::response, mops::methods::update_label>>(web_ctx),
        transformer(
            argument<uid_t>("uid"),
            argument<std::string>("lid"),
            optional_argument<std::string>("name", ""),
            optional_argument<std::string>("color", "")));

    webserver->bind(
        "",
        { "/mops/delete_label" },
        binded_spawn<proxy_to_owner<mops::response, mops::methods::delete_label>>(web_ctx),
        transformer(argument<uid_t>("uid"), argument<std::string>("lid")));

    webserver->bind(
        "",
        { "/mops/create_label_by_symbol" },
        binded_spawn<proxy_to_owner<mops::response, mops::methods::get_or_create_label_by_symbol>>(
            web_ctx),
        transformer(
            argument<uid_t>("uid"),
            argument<std::string>("symbol"),
            optional_argument("force_create", true, validate_as_correct, predefined_value(true))));

    webserver->bind(
        "",
        { "/mops/get_or_create_label_by_symbol" },
        binded_spawn<proxy_to_owner<mops::response, mops::methods::get_or_create_label_by_symbol>>(
            web_ctx),
        transformer(
            argument<uid_t>("uid"),
            argument<std::string>("symbol"),
            optional_argument(
                "force_create", false, validate_as_correct, predefined_value(false))));

    webserver->bind(
        "",
        { "/mops/set_folder_symbol" },
        binded_spawn<proxy_to_owner<mops::response, mops::methods::set_folder_symbol>>(web_ctx),
        transformer(
            argument<uid_t>("uid"),
            argument<fid_t>("fid"),
            optional_argument<std::string>("symbol", "")));

    webserver->bind(
        "",
        { "/mops/set_folders_order" },
        binded_spawn<proxy_to_owner<mops::response, mops::methods::set_folders_order>>(web_ctx),
        transformer(argument<uid_t>("uid"), argument<fid_t>("fid"), argument<fid_t>("prev_fid")));

    webserver->bind(
        "",
        { "/sendbernar/save" },
        binded_spawn<proxy_to_owner<sendbernar::response, sendbernar::methods::save>>(web_ctx),
        transformer(
            argument<uid_t>("uid"),
            argument<fid_t>("fid"),
            optional_argument<mid_t>("old_mid", 0)));

    webserver->bind(
        "",
        { "/sendbernar/send_mail" },
        binded_spawn<proxy_to_owner<sendbernar::response, sendbernar::methods::send_mail>>(web_ctx),
        transformer(
            argument<uid_t>("uid"),
            argument<std::string>("from_email"),
            argument<std::string>("ip"),
            argument<std::string>("request_id"),
            optional_argument<mid_t>("old_mid", 0),
            optional_argument<bool>("notify", 0)));

    webserver->set_custom_key_extractor("", std::bind(&impl::custom_key_extractor, this, ph::_1));
}

void impl::start()
{
    if (lease_node_)
    {
        lease_node_->start_acquire_lease(auth_by_pass_resource_);
    }

    if (rate_limiter_)
    {
        rate_limiter_->run_cleanup_timer();
    }
}

void impl::fini()
{
    lease_node_.reset();
}

void impl::get_auth_master(const node_info_opt_cb& cb)
{
    io_->post([this, self = yplatform::shared_from(this), cb] { cb(master_node_); });
}

void impl::on_busy(
    const std::string& resource,
    const std::string& node_id,
    ymod_lease::ballot_t /*ballot*/,
    const std::string& value)
{
    YLOG_L(info) << "resource for proxy auth_by_password \"" << resource
                 << "\" is busy; owner_id=\"" << node_id << "\", value=\"" << value << "\"";
    if ((!master_node_ || master_node_->id != my_node_.id) && node_id == my_node_.id)
    {
        lease_node_->update_acquire_value(resource, my_node_.to_string());
    }

    if (value.size())
    {
        master_node_ = ymod_mdb_sharder::node_info::from_string(value);
    }
    else
    {
        master_node_ = {};
    }
}

void impl::on_free(const std::string& resource)
{
    YLOG_L(info) << "resource for proxy auth_by_password \"" << resource << "\" is free";
    master_node_ = node_info_opt();
}

void impl::call_limiter(const std::string& key, const std::string& value, const acquire_cb& cb)
{
    if (rate_limiter_)
    {
        return rate_limiter_->try_acquire(key, value, cb);
    }
    cb(error(), acquire_result());
}

std::string impl::custom_key_extractor(stream_ptr stream)
{
    auto current_rps = total_requests_.get_last();
    auto path = stream->request()->url.make_full_path();
    if (path != "/ping" && current_rps > max_rps_)
    {
        WEB_LOG_STREAM(stream, info)
            << "rate limit exceeded current_rps=" << std::to_string(current_rps)
            << " but max_rps=" << std::to_string(max_rps_);
        return "*rate-limit*";
    }
    total_requests_.add(1);
    return path;
}

}

#include <yplatform/module_registration.h>
DEFINE_SERVICE_OBJECT(xeno::web::impl)
