#include "load_user.h"

#include <common/mail_errors.h>
#include <web/mobile/response.h>
#include <xeno/xeno.h>

#include <ymod_mdb_sharder/shards_distributor.h>
#include <ymod_macs/module.h>
#include <yplatform/find.h>

#include <yplatform/yield.h>

namespace xeno::web::internal::methods {

load_user::load_user(web_context_ptr web_ctx, stream_ptr stream, uid_t uid, bool load)
    : web_ctx_(web_ctx), stream_(stream), uid_(uid), load_(load)
{
    macs_ = yplatform::find<ymod_macs::module>("macs");
    shards_distributor_ = yplatform::find<ymod_mdb_sharder::shards_distributor, std::shared_ptr>(
        "shards_distributor");
    web_ = yplatform::find<web::impl, std::shared_ptr>("xeno_web");
    xeno_ = yplatform::find<::xeno::xeno, std::shared_ptr>("xeno");
}

void load_user::operator()(yield_context yield_ctx, error err)
{
    try
    {
        reenter(yield_ctx)
        {
            yield macs_->get_user_shard_id(stream_->ctx(), std::to_string(uid_), yield_ctx);
            if (err)
            {
                code_ = http_code::internal_server_error;
                response_ = std::string("get user shard id error: ") + err.message();
                WEB_LOG(error) << response_;
                yield break;
            }

            yield shards_distributor_->get_owner(stream_->ctx(), shard_id_, yield_ctx);
            if (err)
            {
                code_ = http_code::internal_server_error;
                response_ = "get owner error: " + err.message();
                WEB_LOG(error) << response_;
                yield break;
            }

            if (load_)
            {
                if (owner_.id != shards_distributor_->my_node_id())
                {
                    code_ = http_code::not_acceptable;
                    response_ = "user not owned by this node";
                    WEB_LOG(error) << response_;
                    yield break;
                }

                xeno_->load_user(uid_, shard_id_);
            }
            else
            {
                xeno_->unload_user(uid_);
            }

            code_ = http_code::ok;
            response_ = load_ ? "start loading user" : "start unloading user";
        }
    }
    catch (const std::exception& e)
    {
        WEB_LOG(error) << (load_ ? "load" : "unload") << " user exception: " << e.what();
        code_ = http_code::internal_server_error;
        response_ = e.what();
    }

    if (yield_ctx.is_complete())
    {
        respond(stream_, code_, response_);
    }
}

void load_user::operator()(
    yield_context yield_ctx,
    error err,
    const ymod_mdb_sharder::node_info& owner)
{
    owner_ = owner;
    (*this)(yield_ctx, err);
}

void load_user::operator()(
    yield_context yield_ctx,
    mail_errors::error_code err,
    const shard_id& shard_id)
{
    if (err)
    {
        WEB_LOG(error) << "can't get user shard: " << mail_error_message(err);
        return (*this)(yield_ctx, ::xeno::code::cannot_get_user_shard_id);
    }
    shard_id_ = shard_id;
    (*this)(yield_ctx, {});
}

}

#include <yplatform/unyield.h>
