#include "process_passport_events.h"
#include <web/run_on_node.h>

#include <common/mail_errors.h>
#include <common/msgpack.h>
#include <common/json.h>
#include <xeno/xeno.h>

#include <yplatform/find.h>

#include <yplatform/yield.h>

namespace xeno::web::internal::methods {

process_passport_events::process_passport_events(web_context_ptr web_ctx, stream_ptr stream)
    : web_ctx_(web_ctx), stream_(stream)
{
    macs_ = yplatform::find<ymod_macs::module>("macs");
    xeno_ = yplatform::find<::xeno::xeno, std::shared_ptr>("xeno");
    web_ = yplatform::find<web::impl, std::shared_ptr>("xeno_web");
}

void process_passport_events::operator()(yield_context yield_ctx, error err)
{
    try
    {
        reenter(yield_ctx)
        {
            event_ = unpack_event();
            if (!event_)
            {
                WEB_LOG(error) << "can't unpack event";
                resp_ = make_response(http_code::bad_request);
                yield break;
            }

            if (event_->service != "passport-stream")
            {
                WEB_LOG(error) << "unknown service: " << event_->service;
                // reset subscription
                resp_ = make_response(http_code::reset_content);
                yield break;
            }

            if (event_->operation != "account.changed")
            {
                WEB_LOG(info) << "ignore operation: " << event_->operation;
                yield break;
            }

            if (!parse_event())
            {
                yield break;
            }

            yield macs_->get_user_shard_id(stream_->ctx(), std::to_string(uid_), yield_ctx);
            if (err)
            {
                WEB_LOG(error) << "get user shard id: " << err.message();
                resp_ = make_response(http_code::internal_server_error);
                yield break;
            }

            yield run_on_owner<web::response>(
                shard_id_, stream_, api_type::internal, web_ctx_, yield_ctx);

            yield xeno_->update_user_karma(shard_id_, uid_, karma_, yield_ctx);
            if (err)
            {
                resp_ = make_response(http_code::internal_server_error);
                WEB_LOG(error) << err.message();
            }
        }
    }
    catch (const std::exception& e)
    {
        WEB_LOG(error) << "exception: " << e.what();
        resp_ = make_response(http_code::internal_server_error);
    }

    if (yield_ctx.is_complete())
    {
        respond(stream_, resp_);
    }
}

void process_passport_events::operator()(
    yield_context yield_ctx,
    mail_errors::error_code err,
    const shard_id& shard_id)
{
    if (err)
    {
        WEB_LOG(error) << "can't get user shard: " << mail_error_message(err);
        return (*this)(yield_ctx, ::xeno::code::cannot_get_user_shard_id);
    }

    shard_id_ = shard_id;
    (*this)(yield_ctx, {});
}

xiva_event_opt process_passport_events::unpack_event()
{
    xiva_event event;
    auto request = stream_->request();
    auto message = std::string(request->raw_body.begin(), request->raw_body.end());
    if (unpack(stream_->request()->ctx(), message, event))
    {
        return event;
    }
    return xiva_event_opt();
}

bool process_passport_events::parse_event()
{
    json::value result;
    json::reader reader;
    if (!reader.parse(event_->raw_data, result))
    {
        WEB_LOG(error) << "parse raw data: bad json";
        return false;
    }

    try
    {
        uid_ = std::stoull(result["uid"].asString());
        APPEND_UID_TO_LOG_PREFIX(stream_, uid_);
        karma_.value = std::stoi(result["karma"].asString());
    }
    catch (const std::exception& e)
    {
        WEB_LOG(error) << "parse raw data exception: " << e.what();
        return false;
    }

    return true;
}

}

#include <yplatform/unyield.h>
