#include "auth_by_oauth.h"
#include <auth/social/cache.h>

#include <yplatform/yield.h>

namespace xeno::web::methods {

auth_by_oauth::auth_by_oauth(
    web_context_ptr web_ctx,
    stream_ptr stream,
    const std::string& task_id,
    const std::string& client_id,
    const std::string& client_secret)
    : web_ctx_(web_ctx)
    , stream_(stream)
    , task_id_(task_id)
    , client_id_(client_id)
    , client_secret_(client_secret)
    , is_handle_by_external_token_(false)
{
    ext_mailbox_ = make_external_mailbox();
}

auth_by_oauth::auth_by_oauth(
    web_context_ptr web_ctx,
    stream_ptr stream,
    const std::string& token,
    const std::string& provider,
    const std::string& application,
    const std::string& scope,
    const std::string& client_id,
    const std::string& client_secret)
    : web_ctx_(web_ctx)
    , stream_(stream)
    , token_(token)
    , provider_(provider)
    , application_(application)
    , scope_(scope)
    , client_id_(client_id)
    , client_secret_(client_secret)
    , is_handle_by_external_token_(true)
{
    ext_mailbox_ = make_external_mailbox();
}

void auth_by_oauth::operator()(yield_context yield_ctx, error ec)
{
    if (ec)
    {
        WEB_LOG(error) << ec.message();
        if (ec == ::xeno::code::bad_karma)
        {
            response_ = make_error_response(stream_, ec, "auth error: bad karma");
        }
        else
        {
            response_ = make_error_response(stream_, ec, "auth error: internal backend error");
        }
        respond(stream_, response_);
        return;
    }

    try
    {
        reenter(yield_ctx)
        {
            if (is_handle_by_external_token_)
            {
                yield auth::social::get_refresh_token_by_ext_token(
                    stream_->ctx(),
                    provider_,
                    token_,
                    application_,
                    scope_,
                    web_ctx_->social_settings(),
                    yield_ctx);
            }
            else
            {
                yield auth::social::get_refresh_token_by_task_id(
                    stream_->ctx(), task_id_, web_ctx_->social_settings(), yield_ctx);
            }

            yield auth::social::set_cached_token(
                stream_->ctx(),
                oauth_app_,
                refresh_token_,
                access_token_,
                expires_at_,
                [yield_ctx](error ec) {
                    if (ec)
                    {
                        ec = {}; // do not fail auth on cache errors
                    }
                    yield_ctx(ec);
                });

            endpoints_ = web_ctx_->resolver()->resolve_by_oauth_provider(oauth_provider_);
            if (!endpoints_)
            {
                WEB_LOG(error) << "can't resolve endpoints by oauth provider: " << oauth_provider_;
                response_ = make_error_response(stream_, ec, "auth error: internal backend error");
                yield break;
            }

            account_.imap_ep = endpoints_->imap;
            account_.smtp_ep = endpoints_->smtp;
            update_custom_log_param(stream_, "imap_host", account_.imap_ep.host);

            yield yplatform::spawn(std::make_shared<authorize_user_op>(
                web_ctx_, stream_, ext_mailbox_, account_, client_id_, client_secret_, yield_ctx));

            make_client_response();
        }
    }
    catch (std::exception& e)
    {
        WEB_LOG(error) << "exception " << e.what();
        response_ = make_error_response(stream_, e, "auth error: internal backend error");
    }
    if (yield_ctx.is_complete())
    {
        update_custom_log_param(stream_, "auth_status", (ec ? "error" : "success"));
        respond(stream_, response_);
    }
}

void auth_by_oauth::operator()(
    yield_context yield_ctx,
    error ec,
    const refresh_token_response& refresh_resp)
{
    if (!ec)
    {
        oauth_app_ = refresh_resp.oauth_app;
        oauth_provider_ = refresh_resp.oauth_provider;
        refresh_token_ = refresh_resp.refresh_token;
        access_token_ = refresh_resp.access_token;
        expires_at_ = refresh_resp.expires_at;

        account_.email = refresh_resp.email;
        account_.imap_login = refresh_resp.email;
        account_.smtp_login = refresh_resp.email;

        auth_data data;
        data.type = auth_type::oauth;
        data.imap_credentials = refresh_resp.refresh_token;
        data.smtp_credentials = refresh_resp.refresh_token;
        data.oauth_app = refresh_resp.oauth_app;
        data.uuid = stream_->request()->url.param_value("uuid", "");
        data.last_valid = std::time(nullptr);
        account_.auth_data.push_back(std::move(data));

        update_custom_log_param(stream_, "email", account_.email);
    }
    (*this)(yield_ctx, ec);
}

void auth_by_oauth::operator()(yield_context yield_ctx, error ec, const auth_result& auth_res)
{
    if (!ec)
    {
        auth_res_ = auth_res;
    }
    (*this)(yield_ctx, ec);
}

ext_mailbox_ptr auth_by_oauth::make_external_mailbox()
{
    auto logger = stream_->ctx()->logger();
    logger.append_log_prefix(stream_->ctx()->uniq_id());
    return yplatform::find<xeno>("xeno")->make_external_mailbox(stream_->ctx(), logger);
}

void auth_by_oauth::make_client_response()
{
    json::value resp;
    resp["uid"] = json::value::UInt64(auth_res_.uid);
    resp["is_new_account"] = auth_res_.is_new_account ? "true" : "false";
    resp["email"] = account_.email;
    resp["xtoken"] = auth_res_.x_token;
    response_ = make_ok_response(stream_, resp);
}

}

#include <yplatform/unyield.h>
