#include "auth_by_password.h"
#include <common/crc.h>

#include <web/gendarme.h>
#include <yplatform/yield.h>

#include <set>
#include <map>

namespace xeno::web::methods {

auth_by_password::auth_by_password(
    web_context_ptr web_ctx,
    bool need_resolve,
    stream_ptr stream,
    const std::string& email,
    const std::string& password,
    const std::string& client_id,
    const std::string& client_secret)
    : web_ctx_(web_ctx)
    , need_resolve_(need_resolve)
    , stream_(stream)
    , client_id_(client_id)
    , client_secret_(client_secret)
{
    auth_data auth_data;
    auth_data.type = auth_type::password;
    auth_data.imap_credentials = password;
    auth_data.smtp_credentials = password;
    auth_data.uuid = stream_->request()->url.param_value("uuid", "");
    auth_data.last_valid = std::time(nullptr);

    account_.email = email;
    account_.imap_login = email;
    account_.smtp_login = email;
    account_.auth_data.push_back(std::move(auth_data));

    ext_mailbox_ = make_external_mailbox();

    update_custom_log_param(stream_, "email", email);
    update_custom_log_param(stream_, "pass_crc", std::to_string(crc(password)));
}

auth_by_password::auth_by_password(
    web_context_ptr web_ctx,
    bool need_resolve,
    stream_ptr stream,
    const std::string& email,
    const std::string& imap_login,
    const std::string& imap_password,
    const std::string& imap_host,
    uint16_t imap_port,
    const std::string& imap_ssl,
    const std::string& smtp_login,
    const std::string& smtp_password,
    const std::string& smtp_host,
    uint16_t smtp_port,
    const std::string& smtp_ssl,
    const std::string& client_id,
    const std::string& client_secret)
    : web_ctx_(web_ctx)
    , need_resolve_(need_resolve)
    , stream_(stream)
    , client_id_(client_id)
    , client_secret_(client_secret)
{
    auth_data auth_data;
    auth_data.type = auth_type::password;
    auth_data.imap_credentials = imap_password;
    auth_data.smtp_credentials = smtp_password;
    auth_data.uuid = stream_->request()->url.param_value("uuid", "");
    auth_data.last_valid = std::time(nullptr);
    account_.auth_data.push_back(std::move(auth_data));

    account_.email = email;
    account_.imap_login = imap_login;
    account_.smtp_login = smtp_login;
    account_.imap_ep = { imap_host, imap_port, (imap_ssl == "yes" ? true : false) };
    account_.smtp_ep = { smtp_host, smtp_port, (smtp_ssl == "yes" ? true : false) };

    ext_mailbox_ = make_external_mailbox();

    update_custom_log_param(stream_, "email", email);
    update_custom_log_param(stream_, "imap_host", imap_host);
    update_custom_log_param(stream_, "imap_port", std::to_string(imap_port));
    update_custom_log_param(stream_, "imap_ssl", imap_ssl);
    update_custom_log_param(stream_, "imap_pass_crc", std::to_string(crc(imap_password)));
    update_custom_log_param(stream_, "smtp_host", smtp_host);
    update_custom_log_param(stream_, "smtp_port", std::to_string(smtp_port));
    update_custom_log_param(stream_, "smtp_ssl", smtp_ssl);
    update_custom_log_param(stream_, "smtp_pass_crc", std::to_string(crc(smtp_password)));
}

void auth_by_password::operator()(yield_context yield_ctx, error ec)
{
    try
    {
        reenter(yield_ctx)
        {
            if (need_resolve_)
            {
                yield check_yandex_by_mx(stream_->ctx(), account_.email, yield_ctx);
                if (!ec && exist_yandex_mx_)
                {
                    WEB_LOG(info) << "user has yandex mx entry";
                    provider_ = "yandex";
                    make_client_error_response(::xeno::code::user_from_yandex);
                    yield break;
                }

                yield web_ctx_->resolver()->resolve_by_email_from_ispdb(
                    stream_->ctx(), account_.email, yield_ctx);
                if (ec)
                {
                    yield web_ctx_->resolver()->resolve_by_email_from_cfg(
                        account_.email, yield_ctx);
                }
                if (ec)
                {
                    WEB_LOG(error) << "auth_by_password error: cannot resolve external servers: "
                                   << ec.message();
                    response_ = make_error_response(
                        stream_, ec, "auth error: cannot resolve external servers");
                    yield break;
                }
            }

            yield ext_mailbox_->get_provider(account_.imap_ep, yield_ctx);
            if (ec)
            {
                WEB_LOG(error) << "auth_by_password error: get provider error: " << ec.message();
                make_client_error_response(ec);
                yield break;
            }

            if (provider_ == "yandex")
            {
                WEB_LOG(error) << "user from yandex";
                ec = ::xeno::code::user_from_yandex;
                make_client_error_response(ec);
                yield break;
            }

            if (is_forbidden(provider_))
            {
                WEB_LOG(error) << "forbidden provider " << provider_;
                ec = web_errors::forbidden_provider;
                make_client_error_response(ec);
                yield break;
            }

            yield yplatform::spawn(std::make_shared<authorize_user_op>(
                web_ctx_, stream_, ext_mailbox_, account_, client_id_, client_secret_, yield_ctx));
            if (ec)
            {
                WEB_LOG(error) << ec.message();
                make_client_error_response(ec);
                yield break;
            }

            make_client_response();
        }
    }
    catch (std::exception& e)
    {
        WEB_LOG(error) << "exception " << e.what();
        response_ = make_error_response(stream_, e, "auth error: internal server error");
    }
    if (yield_ctx.is_complete())
    {
        update_custom_log_param(stream_, "auth_status", (ec ? "error" : "success"));
        respond(stream_, response_);
    }
}

void auth_by_password::operator()(yield_context yield_ctx, error ec, const endpoints& result)
{
    if (!ec)
    {
        account_.imap_ep = result.imap;
        account_.smtp_ep = result.smtp;
        update_custom_log_param(stream_, "imap_host", account_.imap_ep.host);
    }

    (*this)(yield_ctx, ec);
}

void auth_by_password::operator()(yield_context yield_ctx, error ec, const auth_result& auth_res)
{
    if (!ec)
    {
        auth_res_ = auth_res;
    }
    (*this)(yield_ctx, ec);
}

void auth_by_password::operator()(yield_context yield_ctx, error ec, const std::string& provider)
{
    if (!ec)
    {
        provider_ = provider;
    }
    (*this)(yield_ctx, ec);
}

void auth_by_password::operator()(yield_context yield_ctx, error ec, bool exist_yandex_mx)
{
    if (!ec)
    {
        exist_yandex_mx_ = exist_yandex_mx;
    }
    (*this)(yield_ctx, ec);
}

void auth_by_password::make_client_response()
{
    json::value resp;
    resp["uid"] = json::value::UInt64(auth_res_.uid);
    resp["is_new_account"] = auth_res_.is_new_account ? "true" : "false";
    resp["email"] = account_.email;
    resp["xtoken"] = auth_res_.x_token;
    response_ = make_ok_response(stream_, resp);
}

void auth_by_password::make_client_error_response(error ec)
{
    if (ec == ::xeno::errc::external_auth_invalid)
    {
        response_ = make_error_response(
            stream_,
            error(web_errors::auth_error, ec.message()),
            "auth error: imap login error",
            (need_resolve_ ? make_servers_hint_json() : Json::objectValue));
    }
    else if (ec == ::xeno::errc::imap_not_connected)
    {
        response_ = make_error_response(
            stream_,
            error(web_errors::auth_error, ec.message()),
            "auth error: cannot connect to imap server",
            (need_resolve_ ? make_servers_hint_json() : Json::objectValue));
    }
    else if (ec == ::xeno::code::imap_protocol_disabled)
    {
        response_ = make_error_response(
            stream_,
            ec,
            "auth error: protocol disabled",
            (need_resolve_ ? make_servers_hint_json() : Json::objectValue));
    }
    else if (ec == ::xeno::code::imap_account_blocked)
    {
        response_ = make_error_response(
            stream_,
            ec,
            "auth error: account blocked",
            (need_resolve_ ? make_servers_hint_json() : Json::objectValue));
    }
    else if (ec == ::xeno::code::smtp_connect_error)
    {
        response_ = make_error_response(
            stream_,
            error(web_errors::auth_error, ec.message()),
            "auth error: cannot connect to smtp server",
            (need_resolve_ ? make_servers_hint_json() : Json::objectValue));
    }
    else if (ec == ::xeno::code::smtp_auth_error)
    {
        response_ = make_error_response(
            stream_,
            error(web_errors::auth_error, ec.message()),
            "auth error: smtp login error",
            (need_resolve_ ? make_servers_hint_json() : Json::objectValue));
    }
    else if (ec == ::xeno::code::smtp_check_email_error)
    {
        response_ = make_error_response(
            stream_,
            ec,
            "auth error: bad email",
            (need_resolve_ ? make_servers_hint_json() : Json::objectValue));
    }
    else if (ec == web_errors::forbidden_provider || ec == ::xeno::code::user_from_yandex)
    {
        response_ = make_error_response(
            stream_, ec, "auth error: forbidden provider", make_provider_json());
    }
    else if (ec == ::xeno::code::server_ip_blacklisted)
    {
        response_ = make_error_response(
            stream_,
            error(web_errors::auth_error, ec.message()),
            "auth error: forbidden server",
            Json::objectValue);
    }
    else if (ec == ::xeno::code::bad_karma)
    {
        response_ = make_error_response(stream_, ec, "auth error: bad karma");
    }
    else if (ec == web_errors::incomplete_params)
    {
        response_ = make_error_response(stream_, ec, "auth error: incomplete smtp params");
    }
    else
    {
        response_ = make_error_response(stream_, ec, "auth error: internal server error");
    }
}

json::value auth_by_password::make_servers_hint_json()
{
    json::value imap_server, smtp_server, result;
    imap_server["host"] = account_.imap_ep.host;
    imap_server["port"] = account_.imap_ep.port;
    imap_server["ssl"] = account_.imap_ep.ssl;
    smtp_server["host"] = account_.smtp_ep.host;
    smtp_server["port"] = account_.smtp_ep.port;
    smtp_server["ssl"] = account_.smtp_ep.ssl;

    result["hint"]["imap_server"] = imap_server;
    result["hint"]["smtp_server"] = smtp_server;
    return result;
}

json::value auth_by_password::make_provider_json()
{
    json::value result;
    result["provider"] = provider_;
    return result;
}

ext_mailbox_ptr auth_by_password::make_external_mailbox()
{
    auto logger = stream_->ctx()->logger();
    logger.append_log_prefix(stream_->ctx()->uniq_id());
    return yplatform::find<xeno>("xeno")->make_external_mailbox(stream_->ctx(), logger);
}

bool auth_by_password::is_forbidden(const std::string& provider)
{
    return web_ctx_->oauth_providers()->count(provider);
}

}

#include <yplatform/unyield.h>
