#pragma once

#include <web/common.h>
#include "../web_context.h"
#include "../auth/get_user_addr.h"
#include <web/mobile/response.h>

#include <auth/authorize.h>
#include <common/mail_errors.h>
#include <common/karma.h>
#include <mdb/accounts_repository.h>
#include <auth/revoke_token.h>

#include <yplatform/find.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>
#include <boost/asio/yield.hpp>

namespace xeno::web::methods {

static const std::string auth_failed_response =
    "PERM_FAIL account_information: 2001, PERM_FAIL get_user_parameters: 2001, PERM_FAIL "
    "yamail_status: 2001, PERM_FAIL settings_setup: 2001, ";

namespace ph = std::placeholders;

template <typename Method>
class auth_protect : boost::asio::coroutine
{
    using string_ptr = std::shared_ptr<std::string>;
    using stream_ptr = ymod_webserver::http::stream_ptr;
    using auth_response_ptr = auth::auth_response_ptr;
    using method = typename std::decay<Method>::type;
    using method_ptr = std::shared_ptr<method>;

public:
    template <typename... Args>
    auth_protect(web_context_ptr web_ctx, stream_ptr stream, Args&&... args)
        : web_ctx_(web_ctx)
        , stream_(stream)
        , xtoken_(std::make_shared<std::string>(get_xtoken(stream_)))
        , real_method_(std::make_shared<method>(web_ctx_, stream_, std::forward<Args>(args)...))
    {
    }

    void operator()(error ec = {}, auth_response_ptr auth_response = nullptr)
    {
        try
        {
            reenter(this)
            {
                if (xtoken_->empty())
                {
                    WEB_LOG(error) << "auth_protect: cannot find x_token";
                    response_ = make_error_response(
                        stream_, web_errors::incomplete_params, "no x-token found");
                    yield break;
                }

                yield auth::authorize(
                    stream_->request()->ctx(),
                    *xtoken_,
                    get_user_ip(stream_),
                    get_user_port(stream_),
                    *this);

                if (ec)
                {
                    WEB_LOG(error) << "auth_protect: blackbox auth: " << ec.message();
                    response_ = make_error_response(
                        stream_, error(web_errors::auth_error, ec.message()), auth_failed_response);
                    yield break;
                }
                auth_response_ = auth_response;
                inject_uid(auth_response->uid);
                WEB_LOG(info) << "auth_protect: success authorization";

                if (is_karma_bad(auth_response_->karma))
                {
                    WEB_LOG(error) << "auth_protect: bad karma";
                    response_ = make_error_response(
                        stream_, code::bad_karma, "auth_protect error: bad karma");
                    yield break;
                }

                yield yplatform::find<ymod_macs::module>("macs")->get_user_shard_id(
                    stream_->ctx(),
                    std::to_string(auth_response_->uid),
                    std::bind(&auth_protect::get_user_shard_cb, *this, ph::_1, ph::_2));
                if (ec)
                {
                    response_ = make_error_response(stream_, ec, "auth error");
                    yield break;
                }

                yield std::make_shared<mdb::accounts_repository>(
                    stream_->ctx(), auth_response_->uid, shard_id_, true)
                    ->get_account(*this);

                if (ec)
                {
                    response_ = make_error_response(stream_, ec, "account not found");
                    yield break;
                }

                inject_user_data(account_);

                if (!has_x_token())
                {
                    yield revoke_token(auth_response_->xtoken_id, *this);
                    if (ec)
                    {
                        WEB_LOG(error) << "auth_protect: revoke xtoken: " << ec.message();
                    }

                    yield revoke_token(auth_response_->token_id, *this);
                    if (ec)
                    {
                        WEB_LOG(error) << "auth_protect: revoke oauth token: " << ec.message();
                    }

                    WEB_LOG(error) << "auth_protect: no auth_data with xtoken_id: "
                                   << auth_response_->xtoken_id;
                    response_ = make_error_response(
                        stream_,
                        error(web_errors::auth_error, "no auth_data with xtoken_id"),
                        auth_failed_response);

                    yield break;
                }

                if (account_.has_security_lock())
                {
                    WEB_LOG(info) << "auth_protect: security lock enabled";
                    response_ = make_error_response(
                        stream_, web_errors::account_not_ready, "account not ready");
                    yield break;
                }

                return (*real_method_)({}, auth_response_);
            }
        }
        catch (const std::exception& e)
        {
            WEB_LOG(error) << "auth_protect: exception:" << e.what();
            response_ = make_error_response(stream_, e, "oauth error: internal backend error");
        }

        if (is_complete())
        {
            return respond(stream_, response_);
        }
    }

    void operator()(error err, const account_t& account)
    {
        if (err)
        {
            WEB_LOG(error) << "auth_protect: " << err.message();
        }
        else
        {
            account_ = account;
        }

        (*this)(err);
    }

    void inject_uid(const uid_t& uid)
    {
        APPEND_UID_TO_LOG_PREFIX(stream_, uid);
        update_custom_log_param(stream_, "uid", std::to_string(uid));
    }

    void get_user_shard_cb(mail_errors::error_code err, const shard_id& shard_id)
    {
        if (err)
        {
            WEB_LOG(error) << "auth_protect: get user shard: " << mail_error_message(err);
            (*this)(::xeno::code::cannot_get_user_shard_id);
        }
        else
        {
            shard_id_ = shard_id;
            (*this)({});
        }
    }

    void inject_user_data(const account_t& account)
    {
        update_custom_log_param(stream_, "imap_host", account.imap_ep.host);
    }

    bool has_x_token() const
    {
        for (auto& data : account_.auth_data)
        {
            if (data.xtoken_id == auth_response_->xtoken_id)
            {
                return true;
            }
        }
        return false;
    }

    template <typename Handler>
    void revoke_token(const std::string& token_id, const Handler& handler)
    {
        auth::revoke_token(stream_->ctx(), token_id, handler);
    }

private:
    web_context_ptr web_ctx_;
    stream_ptr stream_;
    string_ptr xtoken_;
    method_ptr real_method_;

    auth_response_ptr auth_response_;
    account_t account_;
    ::xeno::shard_id shard_id_;

    response response_;
};

}

#include <boost/asio/unyield.hpp>
#include <yplatform/unyield.h>
