#include "delete_items.h"

#include <yplatform/find.h>
#include <yplatform/util/split.h>
#include <boost/asio/yield.hpp>

namespace xeno::web::methods {

namespace {

template <typename T>
std::vector<T> parse_comma_separated_values(const std::string& str)
{
    std::vector<T> res;
    auto splitted = yplatform::util::split(str, ",");
    for (auto& entry : splitted)
    {
        res.push_back(boost::lexical_cast<T>(entry));
    }
    return res;
}

}

delete_items::delete_items(
    web_context_ptr web_ctx,
    stream_ptr stream,
    const mid_vector& mids,
    const tid_vector& tids,
    const fid_t& /*current_folder*/
    )
    : web_ctx_(web_ctx), stream_(stream)
{
    data_->mids = mids;
    data_->tids = tids;
    data_->purge = false;
}

delete_items::delete_items(web_context_ptr web_ctx, stream_ptr stream, bool purge)
    : web_ctx_(web_ctx), stream_(stream)
{
    data_->purge = purge;
    auto req = stream->request();
    auto request = json::from_string(std::string(req->raw_body.begin(), req->raw_body.end()));
    data_->mids = parse_comma_separated_values<mailbox::mid_t>(request.get("mids", "").asString());
    data_->tids = parse_comma_separated_values<mailbox::tid_t>(request.get("tids", "").asString());
}

void delete_items::operator()(error ec, auth_response_ptr auth_resp)
{
    try
    {
        reenter(this)
        {
            data_->uid = auth_resp->uid;
            if (data_->mids.empty() && data_->tids.empty())
            {
                WEB_LOG(error) << "missing mids and tids";
                data_->resp =
                    make_error_response(stream_, web_errors::incomplete_params, "no mids and tids");
                yield break;
            }

            yield(yplatform::find<xeno>("xeno"))
                ->delete_messages(data_->uid, data_->mids, data_->tids, data_->purge, *this);

            if (ec)
            {
                WEB_LOG(error) << ec.message();
                data_->resp = make_error_response(stream_, ec, "delete_items error");
            }
            else
            {
                data_->resp = make_ok_response(stream_);
            }
        }
    }
    catch (std::exception& e)
    {
        WEB_LOG(error) << "exception: " << e.what();
        data_->resp = make_error_response(stream_, e, "delete_items error: internal backend error");
    }

    if (is_complete())
    {
        respond(stream_, data_->resp);
    }
}

}
