#include "messages.h"
#include "proxy.h"

#include <common/json.h>
#include <common/errors.h>

#include <yplatform/find.h>

#include <boost/asio/yield.hpp>

namespace xeno::web::methods {

messages::messages(web_context_ptr web_ctx, stream_ptr stream, bool skip_api_lock)
    : web_ctx_(web_ctx), stream_(stream), check_api_lock_(!skip_api_lock)
{
    xeno_ = yplatform::find<::xeno::xeno, std::shared_ptr>("xeno");
}

void messages::operator()(error ec, auth_response_ptr auth_resp)
{
    try
    {
        reenter(this)
        {
            fids_ = parse_fids();
            if (check_api_lock_ && fids_.size())
            {
                yield xeno_->is_folders_locked_for_api_read(auth_resp->uid, fids_, *this);
                if (ec)
                {
                    WEB_LOG(error) << ec.message();
                    response_ = make_error_response(stream_, ec, "messages error");
                    yield break;
                }

                if (locked_)
                {
                    WEB_LOG(info) << "folder locked for api";
                    response_ =
                        make_error_response(stream_, web_errors::folder_locked, "messages error");
                    yield break;
                }
            }
            proxy(web_ctx_, stream_)();
            need_response_ = false;
        }
    }
    catch (const std::exception& e)
    {
        WEB_LOG(error) << "exception: " << e.what();
        response_ = make_error_response(stream_, e, "messages error: internal backend error");
    }

    if (is_complete() && need_response_)
    {
        respond(stream_, response_);
    }
}

void messages::operator()(error ec, bool locked)
{
    if (!ec)
    {
        locked_ = locked;
    }
    (*this)(ec);
}

fid_vector messages::parse_fids()
{
    auto req = stream_->request();
    json::value result;
    json::reader reader;
    if (!reader.parse(std::string(req->raw_body.begin(), req->raw_body.end()), result))
    {
        throw std::runtime_error("parse parse body: bad json");
    }

    fid_vector fids;
    for (auto& req : result["requests"])
    {
        auto fid = req["fid"];
        if (fid.isNull())
        {
            continue;
        }
        fids.push_back(fid.asString());
    }
    return fids;
}

}
#include <boost/asio/unyield.hpp>
