#include "move_items.h"

#include <yplatform/find.h>
#include <yplatform/util/split.h>
#include <boost/asio/yield.hpp>

namespace xeno::web::methods {

namespace {

std::optional<mailbox::tab_type> get_tab_by_fid(const mailbox::fid_t& fid)
{
    using tab_type = mailbox::tab_type;
    static const std::map<std::string, tab_type> fid_to_tabs = {
        { "-10", tab_type::relevant },
        { "-11", tab_type::news },
        { "-12", tab_type::social },
    };

    auto it = fid_to_tabs.find(fid);
    if (it != fid_to_tabs.end())
    {
        return it->second;
    }
    return std::nullopt;
}

}

move_items::move_items(
    web_context_ptr web_ctx,
    stream_ptr stream,
    const fid_t& dst_fid,
    const mid_vector& mids,
    const tid_vector& tids)
    : web_ctx_(web_ctx), stream_(stream)
{
    data_->mids = mids;
    data_->tids = tids;
    data_->dst_fid = dst_fid;
}

void move_items::operator()(error ec, auth_response_ptr auth_resp)
{
    try
    {
        reenter(this)
        {
            data_->uid = auth_resp->uid;
            if (data_->mids.empty() && data_->tids.empty())
            {
                WEB_LOG(error) << "missing mids and tids";
                data_->resp =
                    make_error_response(stream_, web_errors::incomplete_params, "no mids and tids");
                yield break;
            }

            if (data_->dst_fid.empty())
            {
                WEB_LOG(error) << "missing fid";
                data_->resp = make_error_response(stream_, web_errors::incomplete_params, "no fid");
                yield break;
            }

            if (data_->dst_fid[0] == '-')
            {
                data_->dst_tab = get_tab_by_fid(data_->dst_fid);
            }

            yield(yplatform::find<xeno>("xeno"))
                ->move_messages(
                    data_->uid, data_->mids, data_->tids, data_->dst_fid, data_->dst_tab, *this);

            if (ec)
            {
                WEB_LOG(error) << ec.message();
                data_->resp = make_error_response(stream_, ec, "move_items error");
            }
            else
            {
                data_->resp = make_ok_response(stream_);
            }
        }
    }
    catch (std::exception& e)
    {
        WEB_LOG(error) << "exception: " << e.what();
        data_->resp = make_error_response(stream_, e, "move_items error: internal backend error");
    }

    if (is_complete())
    {
        respond(stream_, data_->resp);
    }
}

}
