#pragma once

#include <web/common.h>
#include "../web_context.h"
#include <web/mobile/response.h>

#include <web/impl.h>
#include <web/rate_limiter.h>

#include <ymod_mdb_sharder/types.h>
#include <yplatform/find.h>
#include <yplatform/yield.h>
#include <yplatform/coroutine.h>

namespace xeno::web::methods {

template <typename Method>
class rate_limit : public std::enable_shared_from_this<rate_limit<Method>>
{
    using stream_ptr = ymod_webserver::http::stream_ptr;
    using method_ptr = std::shared_ptr<Method>;
    using yield_context = yplatform::yield_context<rate_limit>;
    using acquire_result = rate_limiter::acquire_result;

public:
    template <typename... Args>
    rate_limit(
        web_context_ptr web_ctx,
        bool need_resolve,
        stream_ptr stream,
        const std::string& email,
        Args&&... args)
        : web_ctx_(web_ctx)
        , stream_(stream)
        , email_(email)
        , method_(std::make_shared<Method>(
              web_ctx_,
              need_resolve,
              stream_,
              email,
              std::forward<Args>(args)...))
    {
        web_ = yplatform::find<web::impl, std::shared_ptr>("xeno_web");
    }

    void operator()(yield_context yield_ctx, error ec = {})
    {
        try
        {
            reenter(yield_ctx)
            {
                user_ip_ = get_user_ip(stream_);
                yield web_->call_limiter("ip", user_ip_, yield_ctx);
                if (ec)
                {
                    response_ =
                        make_error_response(stream_, ec, "auth error: internal backend error");
                    yield break;
                }
                if (!limiter_result_.success)
                {
                    response_ = make_error_response(
                        stream_,
                        web_errors::rate_limit_exceeded,
                        "auth error: rate limit exceeded",
                        make_limit_expiry_time_hint_json());
                    yield break;
                }

                yield web_->call_limiter("email", email_, yield_ctx);
                if (ec)
                {
                    response_ =
                        make_error_response(stream_, ec, "auth error: internal backend error");
                    yield break;
                }
                if (!limiter_result_.success)
                {
                    response_ = make_error_response(
                        stream_,
                        web_errors::rate_limit_exceeded,
                        "auth error: rate limit exceeded",
                        make_limit_expiry_time_hint_json());
                    yield break;
                }

                return yplatform::spawn(method_);
            }
        }
        catch (std::exception& e)
        {
            WEB_LOG(error) << "rate_limiter exception: " << e.what();
            response_ = make_error_response(stream_, e, "auth error: internal server error");
        }
        if (yield_ctx.is_complete())
        {
            respond(stream_, response_);
        }
    }

    void operator()(yield_context yield_ctx, error ec, const acquire_result& result)
    {
        if (!ec)
        {
            limiter_result_ = result;
        }
        (*this)(yield_ctx, ec);
    }

private:
    json::value make_limit_expiry_time_hint_json()
    {
        json::value result;
        result["hint"]["limit_expiry_time_sec"] =
            time_traits::to_string(limiter_result_.limit_expiry_time);
        return result;
    }

    web_context_ptr web_ctx_;
    stream_ptr stream_;
    std::string email_;
    method_ptr method_;
    std::shared_ptr<web::impl> web_;
    acquire_result limiter_result_;
    response response_;
    std::string user_ip_;
};

}
