#include "response.h"

#include <yplatform/util/sstream.h>

#include <boost/algorithm/string/predicate.hpp>
#include <memory>
#include <sstream>

namespace xeno::web {

namespace {

#define GET_PATH(stream) stream->request()->url.make_full_path()

inline response_type get_response_type(const std::string& path)
{
    using rt = response_type;

    auto truncated_path = boost::starts_with(path, "/api/mobile") ? path.substr(11) : path;
    if (boost::starts_with(truncated_path, "/v2")) return rt::http_code_status;

    static const std::map<std::string, response_type> response_types{
        { "/v1/set_settings", rt::object_unwrapped_status },
        { "/v1/xlist", rt::array_wrapped_status },
        { "/v1/clear_folder", rt::object_unwrapped_status },
        { "/v1/move_to_folder", rt::object_unwrapped_status },
        { "/v1/foo", rt::object_unwrapped_status },
        { "/v1/antifoo", rt::object_unwrapped_status },
        { "/v1/mark_with_label", rt::object_unwrapped_status },
        { "/v1/messages", rt::array_header_status },
        { "/v1/messages_by_timestamp_range", rt::array_header_status },
        { "/v1/mark_read", rt::object_unwrapped_status },
        { "/v1/mark_unread", rt::object_unwrapped_status },
        { "/v1/delete_items", rt::object_unwrapped_status }
    };

    auto it = response_types.find(truncated_path);
    return it != response_types.end() ? it->second : rt::object_wrapped_status;
}

response make_unwrapped_ok_response()
{
    response res;
    json::value resp_data;
    resp_data["status"] = static_cast<int>(status_code::ok);
    res.body = json::to_string(resp_data);
    return res;
}

response make_wrapped_ok_response()
{
    response res;
    json::value resp_data;
    resp_data["status"]["status"] = static_cast<int>(status_code::ok);
    res.body = json::to_string(resp_data);
    return res;
}

void add_wrapped_status_ok(json::value& response)
{
    response["status"]["status"] = static_cast<int>(status_code::ok);
}

void add_unwrapped_status_ok(json::value& response)
{
    response["status"] = static_cast<int>(status_code::ok);
}

response make_http_code_ok_response()
{
    // default respose is fine here
    return {};
}

response make_object_unwrapped_error_response(
    status_code status,
    const std::string& public_error,
    json::value content)
{
    response res;
    content["status"] = static_cast<int>(status);
    content["phrase"] = public_error;
    content["trace"] = "";
    res.body = json::to_string(content);
    res.status = status;
    return res;
}

response make_object_wrapped_error_response(
    status_code status,
    const std::string& public_error,
    json::value content)
{
    response res;
    content["status"]["status"] = static_cast<int>(status);
    content["status"]["phrase"] = public_error;
    content["status"]["trace"] = "";
    res.body = json::to_string(content);
    res.status = status;
    return res;
}

response make_array_wrapped_error_response(
    status_code status,
    const std::string& public_error,
    json::value content)
{
    response res;

    json::value array;
    content["status"]["status"] = static_cast<int>(status);
    content["status"]["phrase"] = public_error;
    content["status"]["trace"] = "";

    array.append(content);
    res.body = json::to_string(array);
    res.status = status;
    return res;
}

response make_array_header_error_response(
    status_code status,
    const std::string&,
    json::value content)
{
    response res;

    json::value header;
    header["error"] = static_cast<int>(status);
    content["header"] = header;

    json::value array;
    array.append(content);
    res.body = json::to_string(array);
    res.status = status;
    return res;
}

response make_http_code_error_response(
    status_code status,
    const std::string& error,
    json::value content)
{
    response res;
    if (status == status_code::tmp_fail)
    {
        res.code = http_code::internal_server_error;
    }
    else
    {
        res.code = http_code::bad_request;
    }

    content["error"] = error;
    content["message"] = "";
    res.body = json::to_string(content);
    res.status = status;
    return res;
}

}

void response::respond(stream_ptr stream, const response& response)
{
    update_custom_log_param(stream, "api_status", to_string(response.status));
    stream->result(response.code, response.body);
}

void response::respond(stream_ptr stream, error err, const std::string& public_error)
{
    auto response = err ? make_ok_response(stream) : make_error_response(stream, err, public_error);
    respond(stream, response);
}

response make_ok_response(stream_ptr stream)
{
    auto type = get_response_type(GET_PATH(stream));
    switch (type)
    {
    case response_type::object_wrapped_status:
        return make_wrapped_ok_response();
    case response_type::object_unwrapped_status:
        return make_unwrapped_ok_response();
    case response_type::http_code_status:
        return make_http_code_ok_response();
    default:
        throw std::runtime_error("unknown response type");
    }
}

response make_ok_response(stream_ptr stream, json::value content)
{
    response res;
    auto type = get_response_type(GET_PATH(stream));
    switch (type)
    {
    case response_type::object_wrapped_status:
        add_wrapped_status_ok(content);
        break;
    case response_type::object_unwrapped_status:
        add_unwrapped_status_ok(content);
        break;
    default:
        throw std::runtime_error("unknown response type");
    }
    res.body = json::to_string(content);
    return res;
}

response make_error_response(
    stream_ptr stream,
    status_code status,
    const std::string& public_error,
    const std::string& private_error,
    json::value content)
{
    response ret;
    auto type = get_response_type(GET_PATH(stream));
    switch (type)
    {
    case response_type::object_wrapped_status:
        ret = make_object_wrapped_error_response(status, public_error, std::move(content));
        break;
    case response_type::object_unwrapped_status:
        ret = make_object_unwrapped_error_response(status, public_error, std::move(content));
        break;
    case response_type::array_wrapped_status:
        ret = make_array_wrapped_error_response(status, public_error, std::move(content));
        break;
    case response_type::array_header_status:
        ret = make_array_header_error_response(status, public_error, std::move(content));
        break;
    case response_type::http_code_status:
        ret = make_http_code_error_response(status, public_error, std::move(content));
        break;
    default:
        throw std::runtime_error("unknown response type");
    }
    ret.private_error = private_error;
    return ret;
}

response make_error_response(
    stream_ptr stream,
    error ec,
    const std::string& public_error,
    json::value content)
{
    using web_errc = web_errors_condition_category;
    if (ec == web_errc::perm_error || ec == errc::perm_error)
    {
        return make_error_response(
            stream,
            status_code::perm_fail,
            public_error.empty() ? "internal error" : public_error,
            ec.message(),
            std::move(content));
    }
    return make_error_response(
        stream,
        status_code::tmp_fail,
        public_error.empty() ? "internal error" : public_error,
        ec.message(),
        std::move(content));
}

response make_error_response(
    stream_ptr stream,
    const std::exception& e,
    const std::string& public_error,
    json::value content)
{
    return make_error_response(
        stream,
        status_code::tmp_fail,
        public_error.empty() ? "internal error" : public_error,
        e.what(),
        std::move(content));
}

void respond(stream_ptr stream, const response& response)
{
    headers_dict headers_copy = response.headers;
    set_mobile_api_status(headers_copy, response.status);
    update_custom_log_param(stream, "api_status", to_string(response.status));
    update_custom_log_param(stream, "reason", response.private_error);
    respond(stream, response.code, response.body, response.content_type, headers_copy);
}

void respond(
    stream_ptr stream,
    http_code ec,
    const std::string& body,
    const std::string& content_type,
    const headers_dict& headers)
{
    stream->set_code(ec);
    stream->set_content_type(content_type);
    for (auto& [name, value] : headers)
    {
        stream->add_header(name, value);
    }
    stream->result_body(body);
}

optional<status_code> get_mobile_api_status(const headers_dict& headers)
{
    auto it = headers.find(MOBILE_API_STATUS_HEADER);
    if (it == headers.end()) return {};
    auto& [name, value] = *it;
    try
    {
        return static_cast<status_code>(boost::lexical_cast<int>(value));
    }
    catch (...)
    {
        return {};
    }
}

void set_mobile_api_status(headers_dict& headers, status_code status)
{
    string code = std::to_string(static_cast<int>(status));
    auto it = headers.find(MOBILE_API_STATUS_HEADER);
    if (it == headers.end())
    {
        headers.emplace(MOBILE_API_STATUS_HEADER, code);
    }
    else
    {
        it->second = code;
    }
}

}
