#pragma once

#include <web/common.h>
#include <web/errors.h>

#include <common/errors.h>
#include <common/json.h>

#include <yplatform/ptree.h>
#include <ymod_httpclient/call.h>

namespace xeno::web {

using http_code = ymod_webserver::codes::code;
using stream_ptr = ymod_webserver::http::stream_ptr;
using headers_dict = ymod_httpclient::headers_dict;

enum class status_code : int
{
    ok = 1,
    tmp_fail,
    perm_fail,
    need_reset
};

struct response
{
    static void respond(stream_ptr stream, const response& response);
    static void respond(stream_ptr stream, error err, const std::string& public_error = "");

    http_code code = http_code::ok;
    std::string body = "{}";
    std::string content_type = "application/json";
    headers_dict headers;
    status_code status = status_code::ok;
    std::string private_error;
};

inline std::string to_string(status_code code)
{
    switch (code)
    {
    case status_code::ok:
        return "ok";
    case status_code::tmp_fail:
        return "tmp_fail";
    case status_code::perm_fail:
        return "perm_fail";
    case status_code::need_reset:
        return "need_reset";
    default:
        return "unknown";
    }
}

// Thanks to wmi team! Few answer types to client is best practice
enum class response_type : int
{
    object_wrapped_status = 0,
    object_unwrapped_status,
    array_wrapped_status,
    array_header_status,
    http_code_status
};

response make_ok_response(stream_ptr stream);
response make_ok_response(stream_ptr stream, json::value content);
response make_error_response(
    stream_ptr stream,
    status_code status,
    const std::string& public_error,
    const std::string& private_error,
    json::value content = Json::objectValue);
response make_error_response(
    stream_ptr stream,
    error ec,
    const std::string& public_error = "",
    json::value content = Json::objectValue);
response make_error_response(
    stream_ptr stream,
    const std::exception& e,
    const std::string& public_error = "",
    json::value content = Json::objectValue);
response make_redirect_response(const std::string& host, uint16_t port);

void respond(stream_ptr stream, const response& response);
void respond(
    stream_ptr stream,
    http_code ec,
    const std::string& body,
    const std::string& content_type = "application/json",
    const headers_dict& headers = headers_dict());

const std::string MOBILE_API_STATUS_HEADER = "x-mmapi-status";

optional<status_code> get_mobile_api_status(const headers_dict& headers);
void set_mobile_api_status(headers_dict& headers, status_code status);

}
