#include "send.h"
#include <web/auth/get_user_addr.h>

#include <mailbox/data_types/message.h>
#include <common/server_info.h>

#include <yplatform/find.h>
#include <yplatform/util/shared_ptr_cast.h>

#include <boost/asio/yield.hpp>

namespace xeno::web::methods {

send::send(web_context_ptr web_ctx, stream_ptr stream) : web_ctx_(web_ctx), stream_(stream)
{
}

void send::operator()(error ec, auth_response_ptr auth_resp)
{
    uid_ = auth_resp->uid;
    user_ticket_ = auth_resp->user_ticket;
    (*this)(ec);
}

void send::operator()(error ec)
{
    try
    {
        reenter(this)
        {
            // make request here, because if json from client is invalid, we will response 500 and
            // client will fall
            send_request_ = make_send_request(stream_);
            log_operation_id(send_request_);
            yield yplatform::find<xeno>("xeno")->compose_and_send(
                uid_, user_ticket_, send_request_, *this);

            if (ec)
            {
                WEB_LOG(error) << ec.message();
                response_ = make_error_response(stream_, ec, "send error: " + ec.message());
            }
            else
            {
                response_ = make_response();
                draft_base_ = parse_draft_base();
            }
            is_already_answered_ = true;
            respond(stream_, response_);

            if (!ec && draft_base_)
            {
                yield yplatform::find<xeno>("xeno")->delete_messages(
                    uid_, { draft_base_ }, {}, true, *this);
                if (ec)
                {
                    WEB_LOG(error) << "delete draft: " << ec.message();
                    // response ok even if error
                }
            }
        }
    }
    catch (std::exception& e)
    {
        WEB_LOG(error) << "exception: " << e.what();
        response_ = make_error_response(stream_, e, "send error: internal backend error");
    }

    if (is_complete() && !is_already_answered_)
    {
        respond(stream_, response_);
    }
}

void send::operator()(error ec, bool duplicated_send)
{
    if (!ec)
    {
        log_duplicated_send(duplicated_send);
    }
    (*this)(ec);
}

send_request_ptr send::make_send_request(stream_ptr stream)
{
    auto req = shared_ptr_cast<std::shared_ptr>::from(stream->request());
    auto headers = send_request::headers_ptr(req, &req->headers);
    auto get_params = send_request::params_ptr(req, &req->url.params);
    auto post_params = std::make_shared<json::value>(
        json::from_string(std::string(req->raw_body.begin(), req->raw_body.end())));
    auto ret = std::make_shared<send_request>(headers, get_params, post_params);

    // override if already in headers
    ret->add_value("x-real-ip", get_user_ip(stream));
    ret->add_value("x-real-port", std::to_string(get_user_port(stream)));
    ret->add_value("x-request-id", stream->ctx()->uniq_id());
    ret->add_value("x-original-host", my_host_name);

    return ret;
}

response send::make_response()
{
    return make_ok_response(stream_);
}

void send::log_operation_id(send_request_ptr request)
{
    update_custom_log_param(stream_, "operation_id", request->get_value("operation_id"));
}

void send::log_duplicated_send(bool duplicated_send)
{
    update_custom_log_param(stream_, "duplicated_send", std::to_string(duplicated_send));
}

mb::mid_t send::parse_draft_base()
{
    mb::mid_t draft_mid{ 0u };
    auto draft_base_ = send_request_->get_value("draft_base");
    if (draft_base_.size())
    {
        // Because sometimes draft_base is fake
        try
        {
            draft_mid = std::stoull(draft_base_);
        }
        catch (const std::exception& e)
        {
            WEB_LOG(info) << "cannot parse draft base: " << draft_base_
                          << " because of exception: " << e.what();
        }
    }
    return draft_mid;
}

}
