#include "store.h"
#include <web/auth/get_user_addr.h>

#include <common/json.h>
#include <mailbox/data_types/message.h>
#include <common/server_info.h>

#include <yplatform/find.h>
#include <yplatform/util/shared_ptr_cast.h>

#include <boost/asio/yield.hpp>

namespace xeno::web::methods {

store::store(web_context_ptr web_ctx, stream_ptr stream) : web_ctx_(web_ctx), stream_(stream)
{
}

void store::operator()(error ec, auth_response_ptr auth_resp)
{
    try
    {
        reenter(this)
        {
            uid_ = auth_resp->uid;
            // make request here, because if json from client is invalid, we will response 500 and
            // client will fall
            store_request_ = make_store_request(stream_);
            yield yplatform::find<xeno>("xeno")->compose_and_save_draft(
                uid_, auth_resp->user_ticket, store_request_, *this);
            if (ec)
            {
                WEB_LOG(error) << ec.message();
                response_ = make_error_response(stream_, ec, "store error: " + ec.message());
            }
            else
            {
                response_ = make_response();
                draft_base = parse_draft_base();
            }

            if (!ec && draft_base)
            {
                yield yplatform::find<xeno>("xeno")->delete_messages(
                    uid_, { draft_base }, {}, true, *this);
                if (ec)
                {
                    WEB_LOG(error) << "delete draft: " << ec.message();
                    // response ok even if error
                }
            }
        }
    }
    catch (std::exception& e)
    {
        WEB_LOG(error) << "exception: " << e.what();
        response_ = make_error_response(stream_, e, "store error: internal backend error");
    }

    if (is_complete())
    {
        respond(stream_, response_);
    }
}

void store::operator()(error ec, json::value response)
{
    if (!ec)
    {
        store_response_ = std::move(response);
    }
    (*this)(ec);
}

store_request_ptr store::make_store_request(stream_ptr stream)
{
    auto req = shared_ptr_cast<std::shared_ptr>::from(stream->request());
    auto headers = store_request::headers_ptr(req, &req->headers);
    auto get_params = store_request::params_ptr(req, &req->url.params);
    auto post_params = std::make_shared<json::value>(
        json::from_string(std::string(req->raw_body.begin(), req->raw_body.end())));
    auto ret = std::make_shared<store_request>(headers, get_params, post_params);

    // override if already in headers
    ret->add_value("x-real-ip", get_user_ip(stream));
    ret->add_value("x-real-port", std::to_string(get_user_port(stream)));
    ret->add_value("x-request-id", stream->ctx()->uniq_id());
    ret->add_value("x-original-host", my_host_name);

    return ret;
}

response store::make_response()
{
    return make_ok_response(stream_, store_response_);
}

mb::mid_t store::parse_draft_base()
{
    mb::mid_t draft_mid{ 0u };
    auto draft_base = store_request_->get_value("draft_base");
    if (draft_base.size())
    {
        // Because sometimes draft_base is fake
        try
        {
            draft_mid = std::stoull(draft_base);
        }
        catch (const std::exception& e)
        {
            WEB_LOG(info) << "cannot parse draft base: " << draft_base
                          << " because of exception: " << e.what();
        }
    }
    return draft_mid;
}

}
