#pragma once

#include <web/common.h>
#include "../web_context.h"
#include <web/mobile/response.h>

#include <auth/authorize.h>
#include <common/errors.h>
#include <xeno/xeno.h>

#include <boost/asio/yield.hpp>

namespace xeno::web::methods {

class sync_status : public boost::asio::coroutine
{
    using stream_ptr = ymod_webserver::http::stream_ptr;
    using auth_response_ptr = auth::auth_response_ptr;
    using sync_status_type = mailbox::sync_status;

public:
    sync_status(web_context_ptr web_ctx, stream_ptr stream) : web_ctx_(web_ctx), stream_(stream)
    {
        xeno_ = yplatform::find<::xeno::xeno, std::shared_ptr>("xeno");
    }

    void operator()(error ec = {}, auth_response_ptr auth_resp = nullptr)
    {
        try
        {
            reenter(this)
            {
                yield xeno_->get_sync_status(auth_resp->uid, *this);
                if (ec)
                {
                    WEB_LOG(error) << ec.message();
                    response_ = make_error_response(stream_, ec, "sync status error");
                    yield break;
                }
                response_ = make_response();
            }
        }
        catch (const std::exception& e)
        {
            WEB_LOG(error) << "exception: " << e.what();
            response_ =
                make_error_response(stream_, e, "sync status error: internal backend error");
        }

        if (is_complete())
        {
            respond(stream_, response_);
        }
    }

    void operator()(error ec, const sync_status_type& status)
    {
        if (!ec)
        {
            status_ = status;
        }
        (*this)(ec);
    }

    response make_response()
    {
        json::value result;
        result["sync_status"] = Json::arrayValue;
        for (auto& [fid, folder_status] : status_.folders)
        {
            json::value status;
            status["fid"] = fid;
            status["external_messages_count"] = folder_status.external_messages_count;
            status["show_spinner"] =
                folder_status.api_read_lock || folder_status.turbo_sync_running;
            result["sync_status"].append(status);
        }

        return make_ok_response(stream_, result);
    }

private:
    web_context_ptr web_ctx_;
    stream_ptr stream_;
    std::shared_ptr<::xeno::xeno> xeno_;
    sync_status_type status_;
    response response_;
};

}
#include <boost/asio/unyield.hpp>
