#pragma once

#include "proxy_settings.h"
#include "auth/get_user_addr.h"

#include <common/http.h>
#include <common/types.h>

#include <ymod_webserver/response.h>
#include <ymod_httpclient/cluster_client.h>

namespace xeno::web {
namespace detail {

inline http::request_t make_proxy_request(stream_ptr stream, proxy_settings_ptr settings)
{
    auto req = stream->request();
    auto& request_line = req->raw_request_line;
    auto parse_request_line = yplatform::util::split(request_line, " ");

    std::string result_url = parse_request_line.at(1);

    std::string headers_str;
    yplatform::sstream headers_stream(headers_str);
    static const std::string USER_IP_HEADER = "X-Real-Ip";
    static const std::string USER_PORT_HEADER = "X-Real-Port";
    headers_stream << USER_IP_HEADER << ": " << get_user_ip(stream) << "\r\n"
                   << USER_PORT_HEADER << ": " << get_user_port(stream) << "\r\n";
    auto& headers = req->headers;
    for (auto& [name, value] : headers)
    {
        if (boost::iequals(name, USER_IP_HEADER)) continue;
        if (boost::iequals(name, USER_PORT_HEADER)) continue;
        if (settings->remove_request_headers.count(name)) continue;
        headers_stream << name << ": " << value << "\r\n";
    }

    if (req->method == ymod_webserver::methods::mth_post)
    {
        return http::request_t::POST(
            result_url, headers_str, std::string(req->raw_body.begin(), req->raw_body.end()));
    }
    else
    {
        return http::request_t::GET(result_url, headers_str);
    }
}

}

using proxy_cb = std::function<void(error, const yhttp::response&)>;

inline void proxy(
    ymod_webserver::http::stream_ptr stream,
    std::shared_ptr<yhttp::cluster_client> client,
    proxy_settings_ptr settings,
    const proxy_cb& cb)
{
    auto ctx = stream->request()->ctx();
    auto request = detail::make_proxy_request(stream, settings);
    client->async_run(ctx, request, [stream, cb, settings](error err, yhttp::response http_resp) {
        if (err)
        {
            return cb(err, {});
        }
        headers_dict headers;
        for (auto& header : settings->proxy_response_headers)
        {
            auto it = http_resp.headers.find(header);
            if (it != http_resp.headers.end())
            {
                headers.emplace(header, it->second);
            }
        }
        respond(stream, http_code(http_resp.status), http_resp.body, "application/json", headers);
        if (auto api_status = get_mobile_api_status(http_resp.headers))
        {
            update_custom_log_param(stream, "api_status", to_string(*api_status));
        }
        cb({}, http_resp);
    });
}

}
