#pragma once

#include <common/types.h>
#include <web/errors.h>

#include <yplatform/log.h>
#include <yplatform/ptree.h>

#include <boost/asio.hpp>
#include <memory>
#include <map>
#include <vector>

namespace xeno::web {

class rate_limiter
    : public std::enable_shared_from_this<rate_limiter>
    , public yplatform::log::contains_logger
{
    struct settings
    {
        int attemps_limit = 10;
        time_traits::duration expire_interval;
    };
    using settings_map = std::map<std::string, settings>;

    struct attemps_info
    {
        int count = 0;
        time_traits::time_point first_attemp_ts;
    };
    using attemps_info_map = std::map<std::string, attemps_info>;
    using argument_attemps_map = std::map<std::string, attemps_info_map>;

public:
    struct acquire_result
    {
        bool success = true;
        time_traits::duration limit_expiry_time;
    };
    using acquire_cb = std::function<void(error, const acquire_result&)>;

    rate_limiter(
        boost::asio::io_service* io,
        const yplatform::ptree& config,
        const yplatform::log::source& logger)
        : yplatform::log::contains_logger(logger), io_(io), cleanup_timer_(*io)
    {
        auto args_cfg = config.equal_range("args");
        for (auto& [tag_name, entry] : boost::make_iterator_range(args_cfg))
        {
            auto name = entry.get<std::string>("name");
            auto attemps_limit = entry.get("attemps_limit", 10);
            auto expire_interval =
                entry.get<time_traits::duration>("expire_interval", time_traits::seconds(600));
            attemps_[name];
            settings_[name] = { attemps_limit, expire_interval };
        }
        cleanup_interval_ =
            config.get<time_traits::duration>("cleanup_interval", time_traits::seconds(3600));
    }

    void try_acquire(const std::string& key, const std::string& value, const acquire_cb& cb)
    {
        io_->post([this, self = shared_from_this(), key, value, cb] {
            auto attemps_it = attemps_.find(key);
            if (attemps_it == attemps_.end())
            {
                YLOG_L(error) << "rate limiter error: argument not found name=" << key;
                return cb(web_errors::rate_limiter_error, acquire_result());
            }

            auto settings_it = settings_.find(key);
            if (settings_it == settings_.end())
            {
                YLOG_L(error) << "rate limiter error: settings for argument not found name=" << key;
                return cb(web_errors::rate_limiter_error, acquire_result());
            }
            auto& settings = settings_it->second;

            acquire_result result;
            auto& attemps = attemps_it->second;
            auto& attemps_info = attemps[value];
            auto now = time_traits::clock::now();
            auto duration = time_traits::duration_cast<time_traits::seconds>(
                now - attemps_info.first_attemp_ts);
            if (duration < settings.expire_interval)
            {
                if (++attemps_info.count > settings.attemps_limit)
                {
                    result.success = false;
                    result.limit_expiry_time = settings.expire_interval - duration;

                    YLOG_L(info)
                        << "rate limiter info: can't acquire because rate limit exсeeded for arg"
                        << " key=" << key << " value=" << value
                        << " attemps_count=" << std::to_string(attemps_info.count)
                        << " limit_expiry_time="
                        << time_traits::to_string(result.limit_expiry_time);
                }
            }
            else
            {
                attemps[value] = { 1, now };
            }
            cb(error(), result);
        });
    }

    void run_cleanup_timer()
    {
        io_->post([this, self = shared_from_this()] {
            YLOG_L(info) << "rate limiter info: run cleanup timer";
            cleanup_timer_.expires_from_now(cleanup_interval_);
            cleanup_timer_.async_wait(std::bind(&rate_limiter::clear_args, shared_from_this()));
        });
    }

    void clear_args()
    {
        io_->post([this, self = shared_from_this()] {
            for (auto& [name, attemps] : attemps_)
            {
                auto settings_it = settings_.find(name);
                if (settings_it == settings_.end())
                {
                    YLOG_L(info) << "rate limiter info: settings for argument not found name="
                                 << name;
                    continue;
                }

                auto& settings = settings_it->second;
                for (auto it = attemps.begin(); it != attemps.end();)
                {
                    auto& [count, attemp_info] = *it;
                    auto first_attemp_ts = attemp_info.first_attemp_ts;
                    auto duration = time_traits::duration_cast<time_traits::seconds>(
                        time_traits::clock::now() - first_attemp_ts);
                    if (duration >= settings.expire_interval)
                    {
                        attemps.erase(it++);
                    }
                    else
                    {
                        ++it;
                    }
                }
            }
            run_cleanup_timer();
        });
    }

    size_t attemps_size(const std::string& key)
    {
        auto it = attemps_.find(key);
        if (it == attemps_.end())
        {
            return 0;
        }
        return it->second.size();
    }

private:
    boost::asio::io_service* io_;
    argument_attemps_map attemps_;
    settings_map settings_;
    time_traits::duration cleanup_interval_;
    time_traits::timer cleanup_timer_;
};

}
