#include "resolver.h"

#include <common/domain.h>

#include <ymod_httpclient/cluster_client.h>
#include <boost/range/adaptor/filtered.hpp>
#include <boost/property_tree/xml_parser.hpp>
#include <iostream>

namespace xeno::web {

resolver::resolver(const yplatform::ptree& resolver)
{
    auto imap_smtp_hosts = resolver.equal_range("imap_smtp_hosts");
    for (auto& [name, entry] : boost::make_iterator_range(imap_smtp_hosts))
    {
        auto imap_cfg = entry.get_child_optional("imap");
        auto smtp_cfg = entry.get_child_optional("smtp");

        if (imap_cfg && smtp_cfg)
        {
            auto imap_ep = get_endpoint(*imap_cfg, default_imap_port);
            auto smtp_ep = get_endpoint(*smtp_cfg, default_smtp_port);

            if (imap_ep && smtp_ep)
            {
                auto endpoints_ptr = std::make_shared<endpoints>(endpoints{ *imap_ep, *smtp_ep });
                auto domains = entry.equal_range("domains");
                for (auto& [name, domain] : boost::make_iterator_range(domains))
                {
                    domains_[domain.data()] = endpoints_ptr;
                }
                auto oauth_provider = entry.get("oauth_provider", "");
                oauth_providers_[oauth_provider] = endpoints_ptr;
            }
        }
    }
}

endpoint_opt resolver::get_endpoint(const yplatform::ptree& endpoint_cfg, uint16_t default_port)
{
    auto host = endpoint_cfg.get("host", "");
    if (!host.empty())
    {
        auto port = endpoint_cfg.get<uint16_t>("port", 0);
        auto ssl = endpoint_cfg.get<bool>("ssl", true);
        if (!port)
        {
            port = default_port;
            ssl = true;
        }
        return endpoint(host, port, ssl);
    }
    return endpoint_opt{};
}

void resolver::resolve_by_email_from_cfg(const std::string& email, const resolve_cb& cb) const
{
    auto domain = get_domain_by_email(email);
    if (domain.empty())
    {
        return cb(web_errors::domain_not_found, endpoints());
    }
    auto it = domains_.find(domain);
    if (it == domains_.end())
    {
        return cb(web_errors::domain_not_found, endpoints());
    }
    cb({}, *it->second);
}

void resolver::resolve_by_email_from_ispdb(
    context_ptr task_ctx,
    const std::string& email,
    const resolve_cb& cb)
{
    auto domain = get_domain_by_email(email);
    if (domain.empty())
    {
        YLOG_CTX(task_ctx->logger(), task_ctx, error) << "resolver error: invalid email";
        return cb(web_errors::domain_not_found, endpoints());
    }
    std::stringstream url;
    url << "/v1.1/" << domain;

    auto request = http::request_t::GET(url.str());
    auto client = yplatform::find<yhttp::cluster_client>("ispdb_client");
    client->async_run(
        task_ctx,
        std::move(request),
        [this, self = shared_from_this(), task_ctx, cb](error err, yhttp::response response) {
            endpoints result;
            try
            {
                if (err)
                {
                    YLOG_CTX(task_ctx->logger(), task_ctx, error)
                        << "resolver error: " << err.message();
                    err = web_errors::ispdb_error;
                }
                else if (response.status / 100 != 2)
                {
                    YLOG_CTX(task_ctx->logger(), task_ctx, error)
                        << "resolver error: " << std::to_string(response.status) << " "
                        << response.reason;
                    err = web_errors::ispdb_error;
                }
                else
                {
                    auto parsed = parse_ispdb_response(response.body);
                    if (!parsed)
                    {
                        YLOG_CTX(task_ctx->logger(), task_ctx, error)
                            << "resolver error: cannot parse endpoints from ispdb";
                        err = web_errors::ispdb_error;
                    }
                    else
                    {
                        result = *parsed;
                    }
                }
            }
            catch (const std::exception& e)
            {
                YLOG_CTX(task_ctx->logger(), task_ctx, error)
                    << "resolver error: exception " << e.what();
                err = error(web_errors::exception_occured, e.what());
            }
            cb(err, result);
        });
}

endpoints_opt resolver::parse_ispdb_response(const std::string& response)
{
    std::stringstream response_str(response);
    yplatform::ptree xml;
    boost::property_tree::xml_parser::read_xml(response_str, xml);

    auto email_provider = xml.get_child_optional("clientConfig.emailProvider");
    if (!email_provider)
    {
        return endpoints_opt();
    }

    auto imap_ep = parse_node(*email_provider, endpoint_type::imap);
    if (!imap_ep)
    {
        return endpoints_opt();
    }
    auto smtp_ep = parse_node(*email_provider, endpoint_type::smtp);
    if (!smtp_ep)
    {
        return endpoints_opt();
    }
    return endpoints{ *imap_ep, *smtp_ep };
}

endpoint_opt resolver::parse_node(const yplatform::ptree& provider, endpoint_type ep_type)
{
    endpoint_opt non_ssl;
    std::string server_name =
        (ep_type == endpoint_type::imap) ? "incomingServer" : "outgoingServer";
    std::string ep_type_name = (ep_type == endpoint_type::imap) ? "imap" : "smtp";
    auto server = provider.equal_range(server_name);
    for (auto& [name, entry] : boost::make_iterator_range(server))
    {
        auto type = entry.get_optional<std::string>("<xmlattr>.type");
        if (!type || *type != ep_type_name)
        {
            continue;
        }
        auto ep = parse_ep(entry);
        if (!ep)
        {
            continue;
        }
        if (ep->ssl)
        {
            return ep;
        }
        non_ssl = ep;
    }
    return non_ssl;
}

endpoint_opt resolver::parse_ep(const yplatform::ptree& server)
{
    auto host = server.get_optional<std::string>("hostname");
    auto port = server.get<uint16_t>("port", 0u);
    auto socket_type = server.get_optional<std::string>("socketType");
    if (!host || host->empty() || !port || !socket_type)
    {
        return endpoint_opt();
    }

    return endpoint{ *host, port, (*socket_type == "SSL" ? true : false) };
};

endpoints_opt resolver::resolve_by_oauth_provider(const std::string& oauth_provider) const
{
    auto it = oauth_providers_.find(oauth_provider);
    if (it == oauth_providers_.end())
    {
        return endpoints_opt();
    }
    return *it->second;
}

}
