#pragma once

#include "errors.h"
#include <common/account.h>
#include <common/http.h>
#include <common/types.h>

#include <yplatform/ptree.h>

#include <exception>
#include <memory>
#include <utility>

namespace xeno::web {

struct endpoints
{
    endpoint imap;
    endpoint smtp;
};
using endpoints_ptr = std::shared_ptr<endpoints>;
using endpoints_opt = boost::optional<endpoints>;

using resolve_cb = std::function<void(error, const endpoints&)>;

class resolver : public std::enable_shared_from_this<resolver>
{
    using domain_endpoints_map = std::map<std::string, endpoints_ptr>;
    using oauth_provider_endpoints_map = std::map<std::string, endpoints_ptr>;
    using imap_host_provider_map = std::map<std::string, std::string>;

public:
    resolver(const yplatform::ptree& cfg);
    void resolve_by_email_from_cfg(const std::string& email, const resolve_cb& cb) const;
    void resolve_by_email_from_ispdb(
        context_ptr task_ctx,
        const std::string& email,
        const resolve_cb& cb);
    endpoints_opt resolve_by_oauth_provider(const std::string& oauth_provider) const;

    endpoint_opt get_endpoint(const yplatform::ptree& endpoint_cfg, uint16_t default_port);

private:
    enum class endpoint_type : int
    {
        imap = 0,
        smtp
    };
    endpoints_opt parse_ispdb_response(const std::string& response);
    endpoint_opt parse_node(const yplatform::ptree& provider, endpoint_type ep_type);
    endpoint_opt parse_ep(const yplatform::ptree& server);

    domain_endpoints_map domains_;
    oauth_provider_endpoints_map oauth_providers_;
    imap_host_provider_map providers_;
    static const uint16_t default_imap_port = 993;
    static const uint16_t default_smtp_port = 465;
};

using resolver_ptr = std::shared_ptr<resolver>;

}
