#pragma once

#include <web/mobile/response.h>
#include <web/impl.h>
#include <web/proxy.h>

#include <ymod_mdb_sharder/shards_distributor.h>
#include <ymod_mdb_sharder/users_distributor.h>
#include <ymod_webserver/response.h>

namespace xeno::web {

namespace ph = std::placeholders;

enum class api_type
{
    mobile,
    internal
};

namespace detail {

inline string make_proxy_url(const string& host, bool use_ssl, int port)
{
    return (use_ssl ? "https://" : "http://") + host + ":" + std::to_string(port);
}

template <typename ClientResponse, typename Handler>
void run_on_node(
    const ymod_mdb_sharder::node_info& node,
    ymod_webserver::http::stream_ptr stream,
    api_type request_api,
    web_context_ptr web_context,
    const Handler& handler)
{
    if (node.id ==
        yplatform::find<ymod_mdb_sharder::shards_distributor>("shards_distributor")->my_node_id())
    {
        handler();
    }
    else
    {
        auto settings = web_context->proxy_settings();
        auto url = request_api == api_type::mobile ?
            make_proxy_url(node.host, settings->mobile_api_use_ssl, settings->mobile_api_port) :
            make_proxy_url(node.host, settings->internal_api_use_ssl, settings->internal_api_port);
        update_custom_log_param(stream, "proxy_to", url);
        auto client = web_context->get_http_client(url);
        web::proxy(stream, client, settings, [stream](error err, const yhttp::response&) {
            if (err)
            {
                WEB_LOG_STREAM(stream, error) << "run_on_node: " << err.message();
                ClientResponse::respond(stream, err, "proxy error");
            }
        });
    }
}

}

template <typename ClientResponse, typename Handler>
inline void run_on_owner(
    const shard_id& shard_id,
    ymod_webserver::http::stream_ptr stream,
    api_type request_api,
    web_context_ptr web_context,
    const Handler& handler)
{
    yplatform::find<ymod_mdb_sharder::shards_distributor>("shards_distributor")
        ->get_owner(
            stream->ctx(),
            shard_id,
            [stream, request_api, web_context, handler](
                error err, const ymod_mdb_sharder::node_info& node) {
                if (err)
                {
                    WEB_LOG_STREAM(stream, error) << "get owner: " << err.message();
                    ClientResponse::respond(stream, err, "get owner error: " + err.message());
                }
                else
                {
                    detail::run_on_node<ClientResponse>(
                        node, stream, request_api, web_context, handler);
                }
            });
}

template <typename ClientResponse, typename Handler>
inline void run_on_owner(
    uid_t uid,
    ymod_webserver::http::stream_ptr stream,
    api_type request_api,
    web_context_ptr web_context,
    const Handler& handler)
{
    yplatform::find<ymod_mdb_sharder::users_distributor>("users_distributor")
        ->get_owner(
            stream->ctx(),
            uid,
            [stream, request_api, web_context, handler](
                error err, const ymod_mdb_sharder::node_info& node) {
                if (err)
                {
                    WEB_LOG_STREAM(stream, error) << "get owner: " << err.message();
                    ClientResponse::respond(stream, err, "get owner error: " + err.message());
                }
                else
                {
                    detail::run_on_node<ClientResponse>(
                        node, stream, request_api, web_context, handler);
                }
            });
}

template <typename ClientResponse, typename Handler>
inline void run_on_auth_master(
    ymod_webserver::http::stream_ptr stream,
    api_type request_api,
    web_context_ptr web_context,
    const Handler& handler)
{
    yplatform::find<web::impl>("xeno_web")
        ->get_auth_master([stream, request_api, web_context, handler](const node_info_opt& master) {
            if (master)
            {
                detail::run_on_node<ClientResponse>(
                    *master, stream, request_api, web_context, handler);
            }
            else
            {
                WEB_LOG_STREAM(stream, error) << "no auth master";
                ClientResponse::respond(stream, web_errors::node_not_found, "no auth master");
            }
        });
}

}
