#pragma once

#include <web/sendbernar/responses.h>
#include <xeno/xeno.h>

#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace xeno::web::sendbernar::methods {

struct send_mail
{
    using yield_context = yplatform::yield_context<send_mail>;
    using stream_ptr = ymod_webserver::http::stream_ptr;
    using addr_vec = std::vector<std::string>;

    send_mail(
        web_context_ptr web_ctx,
        stream_ptr stream,
        uid_t uid,
        const std::string& email,
        const std::string& ip,
        const std::string& request_id,
        mid_t old_mid,
        bool notify)
        : web_ctx(web_ctx)
        , stream(stream)
        , uid(uid)
        , email(email)
        , ip(ip)
        , request_id(request_id)
        , old_mid(old_mid)
        , notify(notify)
    {
    }

    void operator()(yield_context ctx, error ec = {})
    {
        try
        {
            reenter(ctx)
            {
                parse_rcpts();
                yield yplatform::find<xeno>("xeno")->send(
                    uid,
                    email,
                    rcpts,
                    ip,
                    request_id,
                    std::string(
                        stream->request()->raw_body.begin(), stream->request()->raw_body.end()),
                    notify,
                    ctx);

                if (ec)
                {
                    WEB_LOG_STREAM(stream, error) << ec.message();
                    if (ec == ::xeno::code::message_is_spam)
                    {
                        resp = make_error_response(http_code::not_acceptable, ec.message());
                    }
                    else
                    {
                        resp = make_error_response(http_code::internal_server_error, ec.message());
                    }
                    yield break;
                }

                is_already_responded = true;
                resp = make_response();
                response::respond(stream, resp);

                if (old_mid)
                {
                    yield yplatform::find<xeno>("xeno")->delete_messages(
                        uid, { old_mid }, {}, true, ctx);
                    if (ec)
                    {
                        WEB_LOG_STREAM(stream, error) << ec.message();
                        // response ok even if error
                    }
                }
            }
        }
        catch (std::exception& e)
        {
            WEB_LOG_STREAM(stream, error) << e.what();
            resp = make_error_response(http_code::internal_server_error, "internal backend error");
        }

        if (ctx.is_complete() && !is_already_responded)
        {
            response::respond(stream, resp);
        }
    }

    void operator()(yield_context ctx, error ec, mid_t mid)
    {
        this->mid = mid;
        (*this)(ctx, ec);
    }

    response make_response()
    {
        json::value content;
        content["mid"] = mid;
        return make_ok_response(content);
    }

    void parse_rcpts()
    {
        parse_rcpt("to", rcpts);
        parse_rcpt("cc", rcpts);
        parse_rcpt("bcc", rcpts);
    }

    void parse_rcpt(const std::string& key, addr_vec& target)
    {
        auto range = stream->request()->url.params.equal_range(key);
        for (auto& [name, value] : boost::make_iterator_range(range.first, range.second))
        {
            target.push_back(boost::lexical_cast<addr_vec::value_type>(value));
        }
    }

    web_context_ptr web_ctx;
    stream_ptr stream;
    uid_t uid;
    std::string email;
    std::string ip;
    std::string request_id;
    mid_t old_mid;
    bool notify;
    mid_t mid = 0;
    addr_vec rcpts;
    response resp;
    bool is_already_responded = false;
};

}

#include <yplatform/unyield.h>
