#pragma once

#include "resolver.h"
#include "auth/auth_check/auth_check_settings.h"
#include "proxy_settings.h"

#include <auth/social/settings.h>
#include <common/http.h>
#include <mdb/accounts_repository.h>
#include <ymod_httpclient/cluster_client.h>
#include <yplatform/log.h>

#include <memory>

namespace xeno::web {

class web_context
{
public:
    using http_call_ptr = http::http_call_ptr;
    using logger_t = yplatform::log::source;
    using providers = std::set<std::string>;
    using providers_ptr = std::shared_ptr<providers>;

    web_context(
        resolver_ptr resolver,
        auth::social::social_settings_ptr social_settings,
        proxy_settings_ptr proxy_settings,
        auth_check::auth_check_settings_ptr check_settings,
        http_call_ptr call,
        providers_ptr oauth_providers)
        : resolver_(resolver)
        , social_settings_(social_settings)
        , proxy_settings_(proxy_settings)
        , check_settings_(check_settings)
        , call_(call)
        , oauth_providers_(oauth_providers)
    {
        proxy_http_reactor_ = yplatform::find_reactor(proxy_settings_->http_reactor);
    }

    resolver_ptr resolver() const
    {
        return resolver_;
    };

    auth::social::social_settings_ptr social_settings() const
    {
        return social_settings_;
    }

    proxy_settings_ptr proxy_settings() const
    {
        return proxy_settings_;
    }

    auth_check::auth_check_settings_ptr check_settings() const
    {
        return check_settings_;
    }

    std::shared_ptr<yhttp::cluster_client> get_http_client(const std::string& url)
    {
        std::lock_guard<std::mutex> lock(mutex_);
        auto it = http_clients_.find(url);
        if (it != http_clients_.end()) return it->second;

        auto& http_settings = proxy_settings_->http_settings;
        http_settings.nodes.clear();
        http_settings.nodes.emplace_back(url);

        auto client = std::make_shared<yhttp::cluster_client>(*proxy_http_reactor_, http_settings);
        http_clients_[url] = client;
        return client;
    }

    http_call_ptr call() const
    {
        return call_;
    }

    providers_ptr oauth_providers() const
    {
        return oauth_providers_;
    }

private:
    resolver_ptr resolver_;
    auth::social::social_settings_ptr social_settings_;
    yplatform::reactor_ptr proxy_http_reactor_;
    proxy_settings_ptr proxy_settings_;
    auth_check::auth_check_settings_ptr check_settings_;

    std::mutex mutex_;
    std::map<std::string, std::shared_ptr<yhttp::cluster_client>> http_clients_;
    http_call_ptr call_;
    providers_ptr oauth_providers_;
};

using web_context_ptr = std::shared_ptr<web_context>;

}
