#pragma once

#include "synchronization_controller.h"
#include <yplatform/util/safe_call.h>
#include <yplatform/coroutine.h>
#include <yplatform/yield.h>

namespace xeno {

struct dump_controllers_op
{
    using yield_context = yplatform::yield_context<dump_controllers_op>;
    using controller_vector = std::vector<operation_controller_ptr>;
    using io_controller_map = std::map<boost::asio::io_service*, controller_vector>;
    using callback_type = std::function<void(error, const std::vector<controller_dump>&)>;

    context_ptr ctx;
    controller_vector controllers;
    callback_type cb;
    io_controller_map controllers_by_io;
    io_controller_map::iterator it;
    std::vector<controller_dump> result;

    dump_controllers_op(
        context_ptr ctx,
        const controller_vector& controllers,
        const callback_type& cb)
        : ctx(ctx), controllers(controllers), cb(cb)
    {
    }

    void operator()(yield_context yield_ctx)
    {
        reenter(yield_ctx)
        {
            controllers_by_io = group_controllers_by_io(controllers);
            for (it = controllers_by_io.begin(); it != controllers_by_io.end(); ++it)
            {
                yield it->first->post(yield_ctx);
                collect_dumps(it->second, result);
            }
            yplatform::safe_call(ctx, cb, error(), result);
        }
    }

    void operator()(yield_context::exception_type exception)
    {
        try
        {
            std::rethrow_exception(exception);
        }
        catch (std::exception& e)
        {
            LERR_(ctx) << "dump_controllers_op exception: " << e.what();
            cb(code::operation_exception, {});
        }
    }

    io_controller_map group_controllers_by_io(const controller_vector& controllers)
    {
        io_controller_map res;
        for (auto&& controller : controllers)
        {
            res[controller->io()].emplace_back(controller);
        }
        return res;
    }

    void collect_dumps(const controller_vector& controllers, std::vector<controller_dump>& res)
    {
        res.reserve(res.size() + controllers.size());
        for (auto& controller : controllers)
        {
            res.emplace_back(controller->dump());
        }
    }
};

}

#include <yplatform/unyield.h>