#pragma once

#include <common/account.h>
#include <common/karma.h>
#include <mailbox/data_types/cache_mailbox.h>

#include <boost/asio/coroutine.hpp>
#include <boost/asio/yield.hpp>

namespace xeno {

template <typename LocalMb>
class load_cache_op : public boost::asio::coroutine
{
    using handler_type = std::function<void(error, mailbox::cache_mailbox_ptr)>;

public:
    load_cache_op(context_ptr task_ctx, LocalMb local_mailbox, const handler_type& handler)
        : task_ctx_(task_ctx)
        , local_mailbox_(local_mailbox)
        , mailbox_cache_(std::make_shared<mailbox::cache_mailbox>())
        , handler_(handler)
    {
    }

    void operator()(error ec = {})
    {
        try
        {
            reenter(*this)
            {
                mailbox_cache_->logger(task_ctx_->logger());
                yield local_mailbox_->get_account(*this);
                if (ec) yield break;

                if (mailbox_cache_->account().auth_data.empty())
                {
                    ec = code::no_auth_data;
                    yield break;
                }

                yield local_mailbox_->get_karma(mailbox_cache_->account().uid, *this);
                if (is_karma_bad(mailbox_cache_->account().karma))
                {
                    ec = code::bad_karma;
                    yield break;
                }

                yield local_mailbox_->get_folder_vector(*this);
                if (ec) yield break;

                mailbox_cache_->set_initial_folders(folders_);
                mailbox_cache_->update_folders_from_local(folders_);
            }
        }
        catch (const std::exception& e)
        {
            YLOG(task_ctx_->logger(), error) << "cache loader error: exception: " << e.what();
            ec = code::operation_exception;
        }
        if (is_complete())
        {
            handler_(ec, mailbox_cache_);
        }
    }

    void operator()(error error, account_t account)
    {
        if (!error)
        {
            mailbox_cache_->set_account(std::move(account));
        }
        (*this)(error);
    }

    void operator()(error error, mailbox::folder_vector_ptr folders)
    {
        if (!error)
        {
            folders_ = folders;
        }
        (*this)(error);
    }

    void operator()(error error, const karma_t& karma)
    {
        if (!error)
        {
            mailbox_cache_->set_karma(karma);
        }
        (*this)(error);
    }

private:
    context_ptr task_ctx_;
    LocalMb local_mailbox_;
    mailbox::cache_mailbox_ptr mailbox_cache_;
    mailbox::cache_mailbox::path_folder_map::const_iterator folder_it_;
    mailbox::folder_vector_ptr folders_;
    handler_type handler_;
};

}

#include <boost/asio/unyield.hpp>
