#pragma once

#include "iteration_stat.h"
#include "spawn.h"

#include <xeno/rc/external_mailbox.h>
#include <xeno/rc/local_mailbox.h>
#include <xeno/sync_phase.h>
#include <xeno/xeno_settings.h>

#include <common/errors.h>
#include <common/find_deps.h>
#include <mailbox/common.h>
#include <mailbox/data_types/cache_mailbox.h>

#include <yplatform/coroutine.h>
#include <yplatform/log.h>

#include <boost/system/error_code.hpp>
#include <boost/asio/io_service.hpp>
#include <boost/optional.hpp>
#include <memory>
#include <stdexcept>
#include <utility>

namespace xeno {
namespace detail {

template <typename Handler>
class environment_watcher
{
    using handler_t = typename std::decay<Handler>::type;
    using weak_handler_t = std::weak_ptr<handler_t>;

public:
    environment_watcher(iteration_stat_ptr stat, weak_handler_t handler)
        : stat_(stat), weak_handler_(handler)
    {
    }

    environment_watcher(const environment_watcher& other) = delete;

    ~environment_watcher()
    {
        if (num_ != 0)
        {
            auto handler = weak_handler_.lock();
            if (handler)
            {
                handler->handle_operation_finish(stat_, code::environment_destructed_error);
            }
        }
    }

    void inc()
    {
        ++num_;
    }
    void dec()
    {
        --num_;
    }
    void reset()
    {
        num_ = 1;
    }

    int count()
    {
        return num_;
    }

private:
    iteration_stat_ptr stat_;
    weak_handler_t weak_handler_;
    std::atomic_int num_{ 1 };
};

template <typename Handler>
using env_watcher_ptr = std::shared_ptr<environment_watcher<Handler>>;

template <typename SharedPtr, typename Env, typename... Args>
void call_shared(SharedPtr ptr, error err, Env env, Args&&... args)
{
    (*ptr)(err, std::move(env), std::forward<Args>(args)...);
}

template <typename Environment, typename Handler>
class io_wrapper
{
    using env_t = typename std::decay<Environment>::type;
    using handler_t = typename std::decay<Handler>::type;

public:
    io_wrapper(env_t env, Handler&& handler) : env_(env), handler_(std::forward<Handler>(handler))
    {
    }

    template <typename... Args>
    void operator()(Args&&... args)
    {
        auto io = &env_.io;
        auto wrapped = std::bind(handler_, std::move(env_), std::forward<Args>(args)...);
        io->post(wrapped);
    }

private:
    env_t env_;
    handler_t handler_;
};

}

using logger_t = yplatform::log::source;
using logger_ptr = std::shared_ptr<logger_t>;

class multiple_handle_error : public std::runtime_error
{
public:
    multiple_handle_error(const std::string& what) : runtime_error(what)
    {
    }
};

template <typename OperationHandler, typename ExternalMb, typename LocalMb>
class base_environment
{
public:
    using operation_handler_t = typename std::decay<OperationHandler>::type;

    template <typename ParentEnv>
    base_environment(const ParentEnv& env, OperationHandler&& handler)
        : io(env.io)
        , ext_mailbox(env.ext_mailbox)
        , loc_mailbox(env.loc_mailbox)
        , cache_mailbox(env.cache_mailbox)
        , sync_settings(env.sync_settings)
        , ctx(env.ctx)
        , logger(env.logger)
        , typed_logger(env.typed_logger)
        , sync_phase(env.sync_phase)
        , deadline(env.deadline)
        , stat(env.stat)
        , operation_handler_(handler)
    {
    }

    base_environment(
        boost::asio::io_service& io,
        context_ptr ctx,
        const yplatform::log::source& logger,
        iteration_stat_ptr stat,
        OperationHandler&& handler)
        : io(io), ctx(ctx), logger(logger), stat(stat), operation_handler_(handler)
    {
        typed_logger = find_typed_log();
    }

    auto& get_operation_handler()
    {
        return operation_handler_;
    }

    void update_sync_phase(sync_phase::sync_phase_t new_phase)
    {
        stat->count_phase_switch(new_phase);
        sync_phase = new_phase;
    }

    boost::asio::io_service& io;
    ExternalMb ext_mailbox;
    LocalMb loc_mailbox;
    mailbox::cache_mailbox_ptr cache_mailbox;
    synchronization_settings_ptr sync_settings;
    context_ptr ctx;
    yplatform::log::source logger;
    typed_log_ptr typed_logger;
    sync_phase::sync_phase_t sync_phase = sync_phase::initial;
    time_traits::time_point deadline = time_traits::time_point::max();
    iteration_stat_ptr stat;

protected:
    operation_handler_t operation_handler_;

    template <typename U, typename E, typename L>
    friend class base_environment;
};

template <typename OperationHandler, typename ExternalMb, typename LocalMb>
class environment : public base_environment<OperationHandler, ExternalMb, LocalMb>
{
    using parent_t = base_environment<OperationHandler, ExternalMb, LocalMb>;

public:
    template <typename ParentEnv>
    environment(const ParentEnv& env, OperationHandler&& handler)
        : parent_t(env, std::move(handler))
    {
    }

    environment(
        boost::asio::io_service& io,
        context_ptr ctx,
        const yplatform::log::source& logger,
        iteration_stat_ptr stat,
        OperationHandler&& op_handler)
        : parent_t(io, ctx, logger, stat, std::move(op_handler))
    {
    }

    template <typename... Args>
    void operator()(error error, Args&&... args)
    {
        if (handled_)
        {
            throw multiple_handle_error("environment handle called multiple times");
        }

        this->operation_handler_(error, std::forward<Args>(args)...);

        handled_ = true;
    }

    // due to bug in c++ standard
    // http://www.open-std.org/jtc1/sc22/wg21/docs/cwg_active.html#1609
    // it's unclear can we have default parameters before template parameters pack or not
    // so we defined explicit overload to make it clear
    void operator()()
    {
        (*this)(error());
    }

private:
    bool handled_ = false;
};

template <
    typename InterruptHandler,
    typename OperationHandler,
    typename ExternalMb,
    typename LocalMb>
class interruptible_environment : public base_environment<OperationHandler, ExternalMb, LocalMb>
{
    using parent_t = base_environment<OperationHandler, ExternalMb, LocalMb>;
    using this_t =
        interruptible_environment<InterruptHandler, OperationHandler, ExternalMb, LocalMb>;

public:
    using interrupt_handler_t = typename std::decay<InterruptHandler>::type;
    using weak_interrupt_handler_t = std::weak_ptr<interrupt_handler_t>;

    template <typename ParentEnv>
    interruptible_environment(const ParentEnv& env, OperationHandler&& handler)
        : parent_t(env, std::move(handler))
        , weak_interrupt_handler_(env.weak_interrupt_handler_)
        , watcher_(env.watcher_)
    {
        watcher_->inc();
    }

    template <typename Handler>
    interruptible_environment(
        boost::asio::io_service& io,
        context_ptr ctx,
        const yplatform::log::source& logger,
        iteration_stat_ptr stat,
        weak_interrupt_handler_t handler,
        Handler&& op_handler)
        : parent_t(io, ctx, logger, stat, std::forward<Handler>(op_handler))
        , weak_interrupt_handler_(handler)
        , watcher_(
              std::make_shared<detail::environment_watcher<interrupt_handler_t>>(stat, handler))
    {
    }

    template <typename... Args>
    void operator()(error error, Args&&... args)
    {
        if (handled_)
        {
            throw multiple_handle_error("environment handle called multiple times");
        }

        if (auto handler = weak_interrupt_handler_.lock())
        {
            handler->handle_operation_interrupt(error, *this, std::forward<Args>(args)...);
        }

        handled_ = true;
        watcher_->dec();
    }

    // due to bug in c++ standard
    // http://www.open-std.org/jtc1/sc22/wg21/docs/cwg_active.html#1609
    // it's unclear can we have default parameters before template parameters pack or not
    // so we defined explicit overload to make it clear
    void operator()()
    {
        (*this)(error());
    }

    void mark_handled()
    {
        handled_ = true;
        watcher_->reset();
    }

private:
    weak_interrupt_handler_t weak_interrupt_handler_;
    detail::env_watcher_ptr<interrupt_handler_t> watcher_;
    bool handled_ = false;

    template <typename U, typename V, typename E, typename L>
    friend class interruptible_environment;
};

namespace {

struct uninterruptible_t
{
};
uninterruptible_t uninterruptible;

}

template <typename Environment, typename LocalHandler>
auto wrap(Environment&& env, LocalHandler&& handler, uninterruptible_t)
{
    auto wrapped_handler =
        detail::io_wrapper<Environment, LocalHandler>(env, std::forward<LocalHandler>(handler));
    return environment<
        decltype(wrapped_handler),
        decltype(env.ext_mailbox),
        decltype(env.loc_mailbox)>(env, std::move(wrapped_handler));
}

template <
    typename InterruptHandler,
    typename OperationHandler,
    typename ExternalMb,
    typename LocalMb,
    typename LocalHandler>
auto wrap(
    const interruptible_environment<InterruptHandler, OperationHandler, ExternalMb, LocalMb>& env,
    LocalHandler&& handler)
{
    using entironment_t =
        interruptible_environment<InterruptHandler, OperationHandler, ExternalMb, LocalMb>;
    auto wrapped_handler =
        detail::io_wrapper<entironment_t, LocalHandler>(env, std::forward<LocalHandler>(handler));
    return interruptible_environment<
        typename entironment_t::interrupt_handler_t,
        decltype(wrapped_handler),
        ExternalMb,
        LocalMb>(env, std::move(wrapped_handler));
}

template <typename OperationHandler, typename ExternalMb, typename LocalMb, typename LocalHandler>
auto wrap(const environment<OperationHandler, ExternalMb, LocalMb>& env, LocalHandler&& handler)
{
    using entironment_t = environment<OperationHandler, ExternalMb, LocalMb>;
    auto wrapped_handler =
        detail::io_wrapper<entironment_t, LocalHandler>(env, std::forward<LocalHandler>(handler));
    return environment<decltype(wrapped_handler), ExternalMb, LocalMb>(
        env, std::move(wrapped_handler));
}

template <
    typename InterruptHandler,
    typename OperationHandler,
    typename ExternalMb,
    typename LocalMb>
auto make_env(
    boost::asio::io_service* io,
    context_ptr ctx,
    const yplatform::log::source& logger,
    iteration_stat_ptr stat,
    std::weak_ptr<InterruptHandler> handler,
    OperationHandler&& op_handler)
{
    return interruptible_environment<InterruptHandler, OperationHandler, ExternalMb, LocalMb>(
        *io, ctx, logger, stat, handler, std::forward<OperationHandler>(op_handler));
}

template <typename OperationHandler, typename ExternalMb, typename LocalMb>
auto make_env(
    boost::asio::io_service* io,
    context_ptr ctx,
    const yplatform::log::source& logger,
    iteration_stat_ptr stat,
    OperationHandler&& op_handler)
{
    return environment<OperationHandler, ExternalMb, LocalMb>(
        *io, ctx, logger, stat, std::forward<OperationHandler>(op_handler));
}

#define ENV_LOG(env, severity) YLOG(env.logger, severity)
}
