#pragma once

#include "invalidate_auth_data_op.h"

#include <common/errors.h>
#include <xeno/operations/environment.h>

#include <yplatform/yield.h>

namespace xeno {

struct authorize_op
{
    using yield_ctx = yplatform::yield_context<authorize_op>;

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec = {})
    {
        try
        {
            reenter(ctx)
            {
                uid = env.cache_mailbox->account().uid;
                auth_data_copy = env.cache_mailbox->account().auth_data;
                ENV_LOG(env, info) << "reset external mailbox";
                env.ext_mailbox->reset();
                for (it = auth_data_copy.begin();
                     !env.ext_mailbox->authenticated() && it != auth_data_copy.end();
                     ++it)
                {
                    ENV_LOG(env, info) << "get provider from external maibox";
                    yield env.ext_mailbox->get_provider(
                        env.cache_mailbox->account().imap_ep, wrap(env, ctx, uninterruptible));
                    if (ec)
                    {
                        yield break;
                    }

                    if (provider == "yandex")
                    {
                        ENV_LOG(env, info) << "invalidate all auth data because user from yandex";
                        do
                        {
                            yield spawn<invalidate_auth_data_op>(
                                wrap(env, ctx, uninterruptible), *it);
                            ++it;
                        } while (it != auth_data_copy.end());
                        ec = env.cache_mailbox->account().auth_data.size() ?
                            code::user_from_yandex :
                            code::no_auth_data;
                        yield break;
                    }

                    ENV_LOG(env, info) << "external mailbox authorization with xtoken_id";
                    yield env.ext_mailbox->imap_authorize(
                        env.cache_mailbox->account().imap_login,
                        *it,
                        env.cache_mailbox->account().imap_ep,
                        wrap(env, ctx, uninterruptible));

                    last_auth_error = ec;
                    if (!need_increment_tries_count(ec))
                    {
                        yield break;
                    }

                    // we work with copy but we should update original auth_data
                    auth_expired = increment_auth_data_tries_count(env, *it);
                    if (ec == code::imap_response_no && !auth_expired)
                    {
                        ENV_LOG(env, info)
                            << "retry authorization later because error: " << ec.message();
                        last_auth_error = code::auth_tmp_error;
                        continue;
                    }

                    ENV_LOG(env, info) << "invalidate xtoken_id, because of: " << ec.message();
                    yield spawn<invalidate_auth_data_op>(wrap(env, ctx, uninterruptible), *it);
                }
                ec = env.cache_mailbox->account().auth_data.size() ? last_auth_error :
                                                                     code::no_auth_data;
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "authorize op exception: " << e.what();
            ec = code::operation_exception;
        }

        if (ctx.is_complete())
        {
            env(ec);
        }
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec, const std::string& provider)
    {
        if (!ec)
        {
            this->provider = provider;
        }
        (*this)(ctx, std::forward<Env>(env), ec);
    }

    bool need_increment_tries_count(const error& ec)
    {
        if (ec == errc::external_auth_invalid) return true;
        if (ec == code::imap_timeout && provider == "custom") return true;
        return false;
    }

    template <typename Env>
    bool increment_auth_data_tries_count(Env&& env, const auth_data& data_copy)
    {
        auto& account = env.cache_mailbox->account();
        auto it = std::find(account.auth_data.begin(), account.auth_data.end(), data_copy);
        if (it == account.auth_data.end())
        {
            return false;
        }

        ++it->tries_count;
        return is_auth_data_expired(env, *it);
    }

    template <typename Env>
    bool is_auth_data_expired(Env&& env, const auth_data& data)
    {
        switch (data.type)
        {
        case auth_type::password:
            return data.tries_count >= env.sync_settings->background_sync.auth_by_password_tries;
        case auth_type::oauth:
            return data.tries_count >= env.sync_settings->background_sync.auth_by_oauth_tries;
        }
        return false;
    }

    uid_t uid;
    auth_data_vector auth_data_copy;
    auth_data_vector::iterator it;
    error last_auth_error;
    bool auth_expired = false;
    std::string provider;
};

}
#include <yplatform/unyield.h>
