#pragma once

#include <common/errors.h>
#include <mailbox/common.h>
#include <mailbox/data_types/message.h>
#include <mailbox/data_types/folder.h>
#include <xeno/operations/environment.h>

#include <yplatform/log.h>
#include <yplatform/yield.h>

namespace xeno {

struct change_flags_op
{
    using yield_ctx = yplatform::yield_context<change_flags_op>;

    change_flags_op(
        const mailbox::mid_vector& mids,
        const mailbox::flags_t& add_flags,
        const mailbox::flags_t& del_flags)
        : mids_(mids), add_flags_(add_flags), del_flags_(del_flags)
    {
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec = {})
    {
        try
        {
            reenter(ctx)
            {
                yield env.loc_mailbox->get_messages_info_without_flags_by_mid(
                    mids_, wrap(env, ctx, uninterruptible));
                if (ec) yield break;

                messages_by_fid_ = mailbox::message_helpers::split_by_fid(messages_);

                for (current_messages_ = messages_by_fid_.begin();
                     current_messages_ != messages_by_fid_.end();
                     ++current_messages_)
                {
                    current_path_ = env.cache_mailbox->get_path_by_fid(current_messages_->first);
                    ids_ = mailbox::message_helpers::to_imap_id_vector(current_messages_->second);
                    ENV_LOG(env, info)
                        << "marking messages with flags in external mailbox for folder: path="
                        << current_path_.to_string();
                    yield env.ext_mailbox->mark_flags(
                        current_path_,
                        ids_,
                        add_flags_,
                        del_flags_,
                        wrap(env, ctx, uninterruptible));
                    if (ec) yield break;
                }

                ENV_LOG(env, info) << "marking messages with flags in local mailbox";
                yield env.loc_mailbox->change_flags(
                    mids_, add_flags_, del_flags_, wrap(env, ctx, uninterruptible));
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "change flags op exception: " << e.what();
            ec = code::operation_exception;
        }

        if (ctx.is_complete())
        {
            env(ec);
        }
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec, mailbox::message_vector_ptr res)
    {
        if (!ec)
        {
            messages_ = std::move(*res);
        }
        (*this)(ctx, std::forward<Env>(env), ec);
    }

    mailbox::mid_vector mids_;
    mailbox::flags_t add_flags_;
    mailbox::flags_t del_flags_;

    mailbox::message_vector messages_;
    mailbox::fid_messages_map messages_by_fid_;
    mailbox::fid_messages_map::iterator current_messages_;
    mailbox::path_t current_path_;
    mailbox::imap_id_vector_ptr ids_;
};

}

#include <yplatform/unyield.h>
