#pragma once

#include <common/errors.h>
#include <mailbox/common.h>

#include <yplatform/yield.h>
#include <yplatform/log.h>

namespace xeno {

struct check_uidvalidity_op
{
    using message_status = mailbox::message::status_t;
    using yield_ctx = yplatform::yield_context<check_uidvalidity_op>;

    template <typename Env>
    void operator()(
        yield_ctx ctx,
        Env&& env,
        error ec = {},
        mailbox::folder_ptr ext_folder = nullptr)
    {
        try
        {
            reenter(ctx)
            {
                folders = env.cache_mailbox->folders_copy();
                folders_to_check = folders->size();
                for (ifolder = folders->begin(); ifolder != folders->end(); ++ifolder)
                {
                    if (ifolder->status != mailbox::folder::status_t::ok)
                    {
                        --folders_to_check;
                        continue;
                    }
                    yield env.ext_mailbox->get_folder_info(
                        ifolder->path, wrap(env, ctx, uninterruptible));

                    if (ec)
                    {
                        if (ec == errc::imap_not_connected)
                        {
                            // cannot ignore error - reconnect required
                            yield break;
                        }
                        else
                        {
                            ec = code::ok;
                            continue;
                        }
                    }

                    ec = env.cache_mailbox->update_folder_info_from_external(*ext_folder);
                    if (ec)
                    {
                        ++count_bad_uidvalidity;
                    }
                }

                ec = code::ok;
                if (count_bad_uidvalidity)
                {
                    if (count_bad_uidvalidity == folders_to_check)
                    {
                        ec = code::need_clear_mailbox;
                    }
                    else
                    {
                        ec = code::need_check_timestamp;
                    }
                }
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "check uidvalidity error: " << e.what();
            ec = code::operation_exception;
        }

        if (ctx.is_complete())
        {
            return env(ec);
        }
    }

    mailbox::folder_vector_ptr folders;
    mailbox::folder_vector::iterator ifolder;
    size_t folders_to_check = 0;
    size_t count_bad_uidvalidity = 0;
};

}
#include <yplatform/unyield.h>
