#pragma once

#include <common/errors.h>
#include <mailbox/common.h>
#include <xeno/operations/environment.h>

#include <yplatform/log.h>
#include <yplatform/yield.h>

namespace xeno {

struct create_local_folder_op
{
    using yield_ctx = yplatform::yield_context<create_local_folder_op>;
    using folder_t = mailbox::folder;
    using fid_t = mailbox::fid_t;

    create_local_folder_op(const folder_t& folder, const fid_t& parent_fid = "")
        : folder(folder), parent_fid(parent_fid)
    {
    }

    template <typename Env>
    void operator()(
        yield_ctx ctx,
        Env&& env,
        error ec = {},
        mailbox::folder_vector_ptr folders = nullptr)
    {
        try
        {
            reenter(ctx)
            {
                ENV_LOG(env, info)
                    << "creating folder in local mailbox: " << folder.path.to_string();
                yield env.loc_mailbox->create_folder(
                    folder, parent_fid, "", wrap(env, ctx, uninterruptible));
                env.typed_logger->create_folder(
                    env.ctx,
                    env.cache_mailbox->account().uid,
                    folder.path.to_string(),
                    parent_fid,
                    ec);
                if (ec)
                {
                    yield break;
                }

                env.cache_mailbox->update_folders_from_local(folders);
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "create local folder op exception: " << e.what();
            ec = code::operation_exception;
        }

        if (ctx.is_complete())
        {
            env(ec);
        }
    }

    folder_t folder;
    fid_t parent_fid;
};

}
#include <yplatform/unyield.h>
