#pragma once

#include <common/errors.h>
#include <mailbox/common.h>
#include <mailbox/data_types/folder.h>
#include <xeno/operations/environment.h>

#include <yplatform/log.h>
#include <yplatform/yield.h>

namespace xeno {

struct delete_local_folder_op
{
    using yield_ctx = yplatform::yield_context<delete_local_folder_op>;

    delete_local_folder_op(const mailbox::folder& folder) : folder(folder)
    {
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec = {})
    {
        try
        {
            reenter(ctx)
            {
                ENV_LOG(env, info) << "clearing folder in local_mailbox: " << folder.fid;
                yield env.loc_mailbox->clear_folder(folder.fid, wrap(env, ctx, uninterruptible));
                if (ec)
                {
                    yield break;
                }

                if (folder.type == mailbox::folder::type_t::user)
                {
                    ENV_LOG(env, info) << "deleting folder from local_mailbox: " << folder.fid;
                    yield env.loc_mailbox->delete_folder(
                        folder.fid, wrap(env, ctx, uninterruptible));
                }
                else
                {
                    ENV_LOG(env, info) << "deleting mailish folder entry: " << folder.fid;
                    yield env.loc_mailbox->delete_mailish_folder_entry(
                        mailbox::fid_vector{ folder.fid }, wrap(env, ctx, uninterruptible));
                }
                if (ec)
                {
                    yield break;
                }

                env.cache_mailbox->delete_folder_by_fid(folder.fid);
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "delete local folder op exception: " << e.what();
            ec = code::operation_exception;
        }

        if (ctx.is_complete())
        {
            env(ec);
        }
    }

    mailbox::folder folder;
};

}
#include <yplatform/unyield.h>
