#pragma once

#include "move_messages_from_folder_op.h"

#include <common/errors.h>
#include <mailbox/common.h>
#include <mailbox/data_types/message.h>
#include <mailbox/data_types/folder.h>
#include <xeno/operations/environment.h>

#include <yplatform/log.h>
#include <yplatform/yield.h>

namespace xeno {

struct delete_messages_from_folder_op
{
    using yield_ctx = yplatform::yield_context<delete_messages_from_folder_op>;

    delete_messages_from_folder_op(
        const mailbox::folder& folder,
        mailbox::message_vector_ptr messages,
        bool purge = false)
        : folder_(folder), messages_(messages), purge_(purge)
    {
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec = {})
    {
        try
        {
            reenter(ctx)
            {
                trash_folder_ =
                    env.cache_mailbox->get_folder_by_type(mailbox::folder::type_t::trash);

                // The only garanteed way to purge messages is to move them to trash, and then
                // delete
                if (purge_ && trash_folder_)
                {
                    yield spawn<move_messages_from_folder_op>(
                        wrap(env, ctx), folder_, *trash_folder_, std::nullopt, messages_);
                    if (ec) yield break;

                    if (move_result_ && move_result_->size() == messages_->size())
                    {
                        folder_ = *trash_folder_;
                        for (auto& msg : *messages_)
                        {
                            msg.fid = trash_folder_->fid;
                            msg.id = (*move_result_)[msg.id];
                        }
                    }
                    else
                    {
                        // Can't do anything else
                        yield break;
                    }
                }

                if (folder_.type != mailbox::folder::type_t::trash && trash_folder_)
                {
                    yield spawn<move_messages_from_folder_op>(
                        wrap(env, ctx), folder_, *trash_folder_, std::nullopt, messages_);
                    if (ec) yield break;
                }
                else
                {
                    for (auto& msg : *messages_)
                    {
                        imap_ids_->emplace_back(msg.id);
                    }

                    ENV_LOG(env, info) << "expunging messages from=" << folder_.path.to_string();
                    yield env.ext_mailbox->delete_messages(
                        folder_.path, imap_ids_, wrap(env, ctx, uninterruptible));
                    if (ec) yield break;

                    yield env.loc_mailbox->delete_messages_by_id(
                        folder_.fid, *imap_ids_, wrap(env, ctx, uninterruptible));
                    if (ec) yield break;

                    env.cache_mailbox->delete_messages(folder_.path, imap_ids_);
                }
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "delete messages from folder op exception: " << e.what();
            ec = code::operation_exception;
        }

        if (ctx.is_complete())
        {
            env(ec);
        }
    }

    template <typename Env>
    void operator()(
        yield_ctx ctx,
        Env&& env,
        error ec,
        mailbox::imap_id_transform_map_ptr transform_map)
    {
        if (!ec)
        {
            move_result_ = transform_map;
        }
        (*this)(ctx, std::forward<Env>(env), ec);
    }

    mailbox::folder folder_;
    mailbox::message_vector_ptr messages_;
    bool purge_;

    mailbox::folder_opt trash_folder_;
    mailbox::imap_id_transform_map_ptr move_result_;
    mailbox::imap_id_vector_ptr imap_ids_ = std::make_shared<mailbox::imap_id_vector>();
};

}
