#pragma once

#include "delete_messages_from_folder_op.h"

#include <common/errors.h>
#include <mailbox/common.h>
#include <mailbox/data_types/message.h>
#include <mailbox/data_types/folder.h>
#include <xeno/operations/environment.h>

#include <yplatform/log.h>
#include <yplatform/yield.h>

namespace xeno {

struct delete_messages_op
{
    using yield_ctx = yplatform::yield_context<delete_messages_op>;

    delete_messages_op(const mailbox::mid_vector& mids, bool purge)
        : mids_(std::make_shared<mailbox::mid_vector>(mids)), purge_(purge)
    {
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec = {})
    {
        try
        {
            reenter(ctx)
            {
                yield env.loc_mailbox->get_messages_info_without_flags_by_mid(
                    *mids_, wrap(env, ctx, uninterruptible));
                if (ec) yield break;

                messages_by_fid_ = mailbox::message_helpers::split_by_fid(messages_);

                for (current_messages_ = messages_by_fid_.begin();
                     current_messages_ != messages_by_fid_.end();
                     ++current_messages_)
                {
                    {
                        auto current =
                            env.cache_mailbox->get_folder_by_fid(current_messages_->first);
                        if (current)
                        {
                            current_folder_ =
                                std::make_shared<mailbox::folder>(std::move(*current));
                        }
                        else
                        {
                            ec = code::folder_not_found;
                        }
                    }
                    if (ec) yield break;

                    ENV_LOG(env, info) << "deleting messages in single folder: path="
                                       << current_folder_->path.to_string();
                    yield spawn<delete_messages_from_folder_op>(
                        wrap(env, ctx, uninterruptible),
                        *current_folder_,
                        current_messages_->second,
                        purge_);
                    if (ec) yield break;
                }
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "delete messages op exception: " << e.what();
            ec = code::operation_exception;
        }

        if (ctx.is_complete())
        {
            env(ec);
        }
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec, mailbox::message_vector_ptr res)
    {
        if (!ec)
        {
            messages_ = std::move(*res);
        }
        (*this)(ctx, std::forward<Env>(env), ec);
    }

    mailbox::mid_vector_ptr mids_;
    bool purge_;

    mailbox::message_vector messages_;

    mailbox::fid_messages_map messages_by_fid_;
    mailbox::fid_messages_map::iterator current_messages_;
    mailbox::folder_ptr current_folder_;
};

}

#include <yplatform/unyield.h>
