#pragma once

#include <xeno/operations/environment.h>

#include <yplatform/yield.h>

namespace xeno {

struct get_next_sync_oldest_chunk_op
{
    using yield_ctx = yplatform::yield_context<get_next_sync_oldest_chunk_op>;

    get_next_sync_oldest_chunk_op(const mailbox::folder& folder) : folder(folder)
    {
    }

    template <typename Env>
    void operator()(
        yield_ctx ctx,
        Env&& env,
        error ec = {},
        mailbox::message_vector_ptr new_messages = {})
    {
        try
        {
            reenter(ctx)
            {
                processed_range = env.cache_mailbox->get_oldest_processed_range(folder.path);
                if (processed_range.empty())
                {
                    yield get_messages_info(
                        env,
                        folder.count,
                        calc_range_bottom(folder.count, env.sync_settings->oldest_cache_size),
                        wrap(env, ctx));
                    if (ec) yield break;

                    update_range_with_messages(new_messages, env.sync_settings->oldest_cache_size);
                    messages = new_messages;
                }
                else
                {
                    while (messages->empty() && processed_range.bottom() != 1)
                    {
                        range_bottom = calc_range_bottom(
                            processed_range.bottom(), env.sync_settings->oldest_cache_size);
                        yield get_messages_info(
                            env,
                            mailbox::imap_range{ processed_range.bottom(), range_bottom },
                            wrap(env, ctx));
                        if (ec) yield break;

                        processed_range = { processed_range.top(), range_bottom };

                        if (new_messages->empty())
                        {
                            env.cache_mailbox->update_oldest_processed_range(
                                folder.path, processed_range);
                        }
                        else
                        {
                            messages = new_messages;
                            if (messages->front().num == 1)
                            {
                                processed_range = { processed_range.top(), 1 };
                            }
                        }
                    }

                    if (messages->size() < env.sync_settings->oldest_cache_size &&
                        processed_range.bottom() != 1)
                    {
                        yield get_messages_info(
                            env,
                            messages->front().num - 1,
                            calc_range_bottom(
                                messages->front().num - 1, env.sync_settings->oldest_cache_size),
                            wrap(env, ctx));
                        if (ec) yield break;

                        update_range_with_messages(
                            new_messages, env.sync_settings->oldest_cache_size);
                        messages->insert(
                            messages->end(), new_messages->begin(), new_messages->end());
                    }
                }
                std::sort(messages->begin(), messages->end());
                env.cache_mailbox->update_oldest_processed_range(folder.path, processed_range);
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "exception during get_oldest_messages_info_op: " << e.what();
            ec = code::operation_exception;
        }
        if (ctx.is_complete())
        {
            env(ec, messages);
        }
    }

    template <typename Env, typename Handler>
    void get_messages_info(Env&& env, const mailbox::imap_range& range, Handler&& h)
    {
        ENV_LOG(env, info) << "getting message list by id from external mailbox: "
                           << folder.path.to_string() << " [" << range.top() << ", "
                           << range.bottom() << "]";
        env.ext_mailbox->get_messages_info_by_id(folder.path, range, std::forward<Handler>(h));
    }

    template <typename Env, typename Handler>
    void get_messages_info(Env&& env, mailbox::num_t start, mailbox::num_t end, Handler&& h)
    {
        ENV_LOG(env, info) << "getting message list by num from external mailbox: "
                           << folder.path.to_string() << " [" << start << ", " << end << "]";
        env.ext_mailbox->get_messages_info_by_num(
            folder.path, start, end, std::forward<Handler>(h));
    }

    void update_range_with_messages(
        mailbox::message_vector_ptr new_messages,
        mailbox::num_t chunk_size)
    {
        auto new_top = processed_range.top();
        if (!new_top)
        {
            new_top = folder.uidnext - 1;
        }

        if (new_messages->size() < chunk_size)
        {
            processed_range = mailbox::imap_range{ new_top, 1 };
        }
        else
        {
            processed_range = mailbox::imap_range{
                new_top, std::min_element(new_messages->begin(), new_messages->end())->id
            };
        }
    }

    mailbox::imap_id_t calc_range_bottom(mailbox::imap_id_t top_id, mailbox::num_t cache_size)
    {
        return top_id >= cache_size ? top_id - cache_size + 1 : 1;
    }

    mailbox::folder folder;
    mailbox::imap_range processed_range;
    mailbox::message_vector_ptr messages = std::make_shared<mailbox::message_vector>();

    mailbox::imap_id_t range_bottom;
};

}

#include <yplatform/unyield.h>
