#pragma once

#include <auth/revoke_token.h>
#include <common/errors.h>
#include <xeno/operations/environment.h>
#include <xiva/xiva.h>

#include <yplatform/yield.h>

namespace xeno {

struct invalidate_auth_data_op
{
    using yield_ctx = yplatform::yield_context<invalidate_auth_data_op>;

    invalidate_auth_data_op(const auth_data& auth_data) : auth_data_(auth_data)
    {
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec = {})
    {
        if (ec)
        {
            ENV_LOG(env, error) << "invalidate auth data op error: " << ec.message();
            return env(ec);
        }

        try
        {
            reenter(ctx)
            {
                uid_ = env.cache_mailbox->account().uid;
                yield auth::revoke_token(
                    env.ctx, auth_data_.xtoken_id, wrap(env, ctx, uninterruptible));

                if (auth_data_.uuid.size())
                {
                    yield unsubscribe(
                        uid_,
                        auth_data_.uuid,
                        env.ctx,
                        env.sync_settings->xiva_settings,
                        wrap(env, ctx, uninterruptible));
                }

                yield env.loc_mailbox->invalidate_auth_data(
                    auth_data_.xtoken_id, wrap(env, ctx, uninterruptible));

                env.cache_mailbox->invalidate_auth_data(auth_data_);
                env.ext_mailbox->reset();
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "invalidate auth data op exception: " << e.what();
            ec = code::operation_exception;
        }

        if (ctx.is_complete())
        {
            env(ec);
        }
    }

    uid_t uid_;
    auth_data auth_data_;
};

}
#include <yplatform/unyield.h>
