#pragma once

#include <common/errors.h>
#include <mailbox/common.h>
#include <mailbox/data_types/sync_newest_state.h>
#include <xeno/operations/environment.h>

#include <yplatform/log.h>
#include <yplatform/yield.h>

namespace xeno {

struct load_folder_top_op
{
    using message_status = mailbox::message::status_t;
    using yield_ctx = yplatform::yield_context<load_folder_top_op>;

    load_folder_top_op(const mailbox::folder& folder) : folder(folder)
    {
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec = {})
    {
        try
        {
            reenter(ctx)
            {
                yield env.loc_mailbox->get_messages_info_top(
                    folder.fid, env.sync_settings->newest_count, wrap(env, ctx));
                if (ec)
                {
                    yield break;
                }

                mailbox::imap_id_message_map msg_top;
                for (auto& msg : *top_messages)
                {
                    msg_top[msg.id] = msg;
                    msg_top[msg.id].status =
                        (msg.mid ? message_status::ok : message_status::to_download_body);
                }

                if (top_messages->size() == env.sync_settings->newest_count)
                {
                    env.cache_mailbox->get_folder(folder.path).api_read_lock = false;
                }

                state = env.cache_mailbox->sync_newest_state();
                state->folders.insert({ folder.path, { msg_top } });
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "exception during load_folder_top: " << e.what();
            ec = code::operation_exception;
        }
        if (ctx.is_complete())
        {
            env(ec);
        }
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec, mailbox::message_vector_ptr messages)
    {
        top_messages = messages;
        (*this)(ctx, std::forward<Env>(env), ec);
    }

    mailbox::sync_newest_state_ptr state;
    mailbox::message_vector_ptr top_messages;
    mailbox::folder folder;
};

}
#include <yplatform/unyield.h>
