#pragma once

#include "authorize_op.h"
#include "sync_folders_op.h"
#include "sync_newest_op.h"
#include "sync_oldest_op.h"
#include "sync_oldest_flags_and_deleted_op.h"
#include "redownload_messages_op.h"

#include <common/errors.h>
#include <common/types.h>
#include <mailbox/common.h>
#include <mailbox/external/statistics.h>
#include <xeno/operations/environment.h>

#include <yplatform/yield.h>

#include <pa/async.h>

namespace xeno {

struct main_op
{
    using yield_ctx = yplatform::yield_context<main_op>;

    template <typename Environment>
    void operator()(yield_ctx ctx, Environment&& env, error ec = {})
    {
        if (ec)
        {
            if (security_lock)
            {
                ENV_LOG(env, info) << "first sync error: " << ec.message();
            }
            return env(ec);
        }

        try
        {
            reenter(ctx)
            {
                security_lock = env.cache_mailbox->account().has_security_lock();
                if (security_lock)
                {
                    first_sync_start = time_traits::clock::now();

                    ENV_LOG(env, info) << "authorizing (first sync)";
                    // Wraps uninterruptible to prevent permanent iteration deadline before erasing
                    // security lock
                    yield spawn<authorize_op>(wrap(env, ctx, uninterruptible));

                    ENV_LOG(env, info) << "syncing folders (first sync)";
                    env.update_sync_phase(sync_phase::sync_folders);
                    yield spawn<sync_folders_op>(wrap(env, ctx, uninterruptible));

                    ENV_LOG(env, info) << "erasing security locks";
                    yield spawn<erase_security_locks_op>(wrap(env, ctx));

                    inbox_path =
                        env.cache_mailbox->get_path_by_folder_type(mailbox::folder::type_t::inbox);
                    if (!inbox_path)
                    {
                        ENV_LOG(env, error) << "main op error: inbox path not found";
                        ec = code::folder_not_found;
                        yield break;
                    }

                    ENV_LOG(env, info)
                        << "syncing newest messages from " << inbox_path->to_string();
                    env.update_sync_phase(sync_phase::sync_newest);
                    yield spawn<sync_newest_op>(
                        wrap(env, ctx), mailbox::path_vector{ *inbox_path });
                    log_first_sync_duration(env.cache_mailbox->account().uid);

                    paths_to_sync = get_paths(env.cache_mailbox->folders());
                    paths_to_sync.erase(
                        std::remove(paths_to_sync.begin(), paths_to_sync.end(), *inbox_path),
                        paths_to_sync.end());
                }
                else
                {
                    if (!env.ext_mailbox->authenticated())
                    {
                        ENV_LOG(env, info) << "authorizing";
                        yield spawn<authorize_op>(wrap(env, ctx));
                    }

                    ENV_LOG(env, info) << "syncing folders";
                    env.update_sync_phase(sync_phase::sync_folders);
                    yield spawn<sync_folders_op>(wrap(env, ctx));

                    paths_to_sync = get_paths(env.cache_mailbox->folders());
                }

                ENV_LOG(env, info)
                    << "syncing newest messages" << (security_lock ? " from other folders" : "");
                env.update_sync_phase(sync_phase::sync_newest);
                yield spawn<sync_newest_op>(wrap(env, ctx), paths_to_sync);

                ENV_LOG(env, info) << "syncing older messages";
                env.update_sync_phase(sync_phase::sync_oldest);
                yield spawn<sync_oldest_op>(wrap(env, ctx));

                ENV_LOG(env, info) << "syncing flags and deletions";
                env.update_sync_phase(sync_phase::sync_oldest_flags_and_deleted);
                yield spawn<sync_oldest_flags_and_deleted_op>(
                    wrap(env, ctx), env.sync_settings->oldest_flags_and_deletions_chunk_size);

                ENV_LOG(env, info) << "redownloading failed messages";
                env.update_sync_phase(sync_phase::redownload_messages);
                yield spawn<redownload_messages_op>(wrap(env, ctx));
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "main op exception: " << e.what();
            ec = code::operation_exception;
        }

        if (ctx.is_complete())
        {
            env(ec);
        }
    }

    void log_first_sync_duration(uid_t uid)
    {
        auto first_sync_duration = time_traits::duration_cast<time_traits::milliseconds>(
            time_traits::clock::now() - first_sync_start);
        pa::async_profiler::add(
            pa::xeno,
            "main_op",
            "first_sync",
            std::to_string(uid),
            static_cast<uint32_t>(first_sync_duration.count()));
    }

    mailbox::path_vector get_paths(const mailbox::cache_mailbox::path_folder_map& folders)
    {
        mailbox::path_vector paths;
        for (auto& [path, folder] : folders)
        {
            paths.emplace_back(folder.path);
        }
        return paths;
    }

    bool security_lock = false;
    mailbox::path_vector paths_to_sync;
    mailbox::path_t_opt inbox_path;
    time_traits::time_point first_sync_start;
};

}
#include <yplatform/unyield.h>
