#pragma once

#include <common/errors.h>
#include <mailbox/common.h>
#include <mailbox/data_types/message.h>
#include <mailbox/data_types/folder.h>
#include <xeno/operations/environment.h>

#include <yplatform/log.h>
#include <yplatform/yield.h>

namespace xeno {

struct move_messages_from_folder_op
{
    using yield_ctx = yplatform::yield_context<move_messages_from_folder_op>;

    move_messages_from_folder_op(
        const mailbox::folder& folder_from,
        const mailbox::folder& folder_to,
        const mailbox::tab_opt& tab_to,
        mailbox::message_vector_ptr messages)
        : folder_from_(folder_from), folder_to_(folder_to), tab_to_(tab_to), messages_(messages)
    {
        for (auto& msg : *messages_)
        {
            external_move_coords_->emplace_back(msg.id);
            local_move_coords_.emplace_back(mailbox::move_coordinates{ msg.id, 0, msg.mid });
        }
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec = {})
    {
        try
        {
            reenter(ctx)
            {
                ENV_LOG(env, info) << "moving messages from=" << folder_from_.path.to_string()
                                   << ", to=" << folder_to_.path.to_string();
                yield env.ext_mailbox->move_messages(
                    folder_from_.path,
                    folder_to_.path,
                    external_move_coords_,
                    wrap(env, ctx, uninterruptible));
                if (ec)
                {
                    yield break;
                }

                if (!moved_ids_ || moved_ids_->size() != external_move_coords_->size())
                {
                    yield env.loc_mailbox->delete_messages_by_id(
                        folder_from_.fid, *external_move_coords_, wrap(env, ctx, uninterruptible));
                    if (ec)
                    {
                        yield break;
                    }

                    env.cache_mailbox->delete_messages(folder_from_.path, external_move_coords_);
                }
                else
                {
                    for (auto& coords : local_move_coords_)
                    {
                        coords.new_imap_id = (*moved_ids_)[coords.cur_imap_id];
                    }

                    yield env.loc_mailbox->move_messages(
                        folder_from_.fid,
                        folder_to_.fid,
                        tab_to_,
                        local_move_coords_,
                        wrap(env, ctx, uninterruptible));
                    if (ec)
                    {
                        yield break;
                    }

                    env.cache_mailbox->move_messages(
                        folder_from_.path, folder_to_.path, moved_ids_);
                }
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "move messages from folder op exception: " << e.what();
            ec = code::operation_exception;
        }

        if (ctx.is_complete())
        {
            env(ec, moved_ids_);
        }
    }

    template <typename Env>
    void operator()(
        yield_ctx ctx,
        Env&& env,
        error ec,
        mailbox::imap_id_transform_map_ptr moved_ids)
    {
        if (!ec)
        {
            moved_ids_ = moved_ids;
        }
        (*this)(ctx, std::forward<Env>(env), ec);
    }

    mailbox::folder folder_from_;
    mailbox::folder folder_to_;
    mailbox::tab_opt tab_to_;
    mailbox::message_vector_ptr messages_;

    mailbox::imap_id_vector_ptr external_move_coords_ = std::make_shared<mailbox::imap_id_vector>();
    mailbox::move_coordinates_vec local_move_coords_;

    mailbox::imap_id_transform_map_ptr moved_ids_;
};

}

#include <yplatform/unyield.h>
