#pragma once

#include "sync_message_op.h"
#include "update_redownload_messages_cache_op.h"

#include <common/errors.h>
#include <common/random.h>
#include <mailbox/data_types/redownload_messages_state.h>
#include <xeno/operations/environment.h>

#include <yplatform/log.h>
#include <yplatform/yield.h>

#include <memory>

namespace xeno {

// split messages by errors_count into groups with max delay defined for every group
// delay between two failed downloads is being chosen randomly between 0 and current group max delay
// successful downloading continues without delay
struct redownload_messages_op
{
    using yield_ctx = yplatform::yield_context<redownload_messages_op>;

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec = {})
    {
        try
        {
            reenter(ctx)
            {
                state = env.cache_mailbox->redownload_messages_state();
                if (state->has_new_failures ||
                    (state->messages->empty() && state->has_another_messages_in_local_mb))
                {
                    yield spawn<update_redownload_messages_cache_op>(wrap(env, ctx));
                    if (ec) yield break;

                    update_current_delay_and_group(env.sync_settings);
                }

                if (time_traits::clock::now() <
                    state->last_message_download_time + state->current_delay)
                {
                    yield break;
                }

                while (state->messages->size())
                {
                    message = state->messages->back();
                    if (message.errors_count > state->max_error_count_on_last_cache_update &&
                        state->has_another_messages_in_local_mb)
                    {
                        yield spawn<update_redownload_messages_cache_op>(wrap(env, ctx));
                        if (ec) yield break;

                        continue;
                    }
                    if (message.errors_count > state->current_group.errors_count)
                    {
                        update_current_delay_and_group(env.sync_settings);
                        yield break;
                    }

                    state->last_message_download_time = time_traits::clock::now();
                    path = env.cache_mailbox->get_path_by_fid(message.fid);
                    ENV_LOG(env, info)
                        << "retrying download: fid=" << message.fid << ", imap_id=" << message.id;
                    yield spawn<sync_message_op>(
                        wrap(env, ctx),
                        path,
                        message.id,
                        "redownload_messages",
                        message,
                        mailbox::notification_type::disabled);

                    if (ec)
                    {
                        {
                            auto it = state->find_message(message.id, message.fid);
                            if (it != state->messages->end())
                            {
                                it->errors_count = message.errors_count;
                                it->saved_errors_count = message.saved_errors_count;
                                std::sort(
                                    state->messages->begin(),
                                    state->messages->end(),
                                    [](const mailbox::message& lhs, const mailbox::message& rhs) {
                                        return lhs.errors_count > rhs.errors_count;
                                    });
                            }
                        }
                        update_current_delay_and_group(env.sync_settings);
                        yield break;
                    }
                    {
                        auto it = state->find_message(message.id, message.fid);
                        if (it != state->messages->end())
                        {
                            state->messages->erase(it);
                        }
                    }
                }
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "exception during redownload_messages: " << e.what();
            ec = code::operation_exception;
        }
        if (ctx.is_complete())
        {
            env(ec);
        }
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec, const mailbox::message& msg)
    {
        message = msg;
        (*this)(ctx, env, ec);
    }

    void update_current_delay_and_group(synchronization_settings_ptr sync_settings)
    {
        if (state->messages->size())
        {
            auto it = sync_settings->redownload_messages_delay_groups.lower_bound(
                state->messages->back().errors_count);
            // redownload_messages_delays contains uint32::max so iterator is always valid
            state->current_group.errors_count = it->first;
            state->current_group.delay = it->second;
            state->current_delay = time_traits::seconds(
                random_int(0ll, time_traits::get_seconds_count(state->current_group.delay)));
        }
    }

    mailbox::message message;
    mailbox::path_t path;

    mailbox::redownload_messages_state_ptr state;
};

}
#include <yplatform/unyield.h>
