#pragma once

#include "list_status_op.h"
#include "check_uidvalidity_op.h"
#include "create_local_folder_op.h"
#include "create_external_folder_op.h"
#include "update_and_clear_local_folder_op.h"
#include "delete_local_folder_op.h"
#include "check_timestamps_op.h"
#include "clear_mailbox_op.h"
#include "erase_security_locks_op.h"

#include <common/errors.h>
#include <mailbox/common.h>
#include <xeno/operations/environment.h>

#include <yplatform/log.h>
#include <yplatform/yield.h>

#include <memory>

namespace xeno {

struct sync_folders_op
{
    using yield_ctx = yplatform::yield_context<sync_folders_op>;
    using folder_status = mailbox::folder::status_t;

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec = {})
    {
        try
        {
            reenter(ctx)
            {
                yield spawn<list_status_op>(wrap(env, ctx, uninterruptible));
                if (ec)
                {
                    yield break;
                }

                security_lock = env.cache_mailbox->account().has_security_lock();
                if (security_lock)
                {
                    ENV_LOG(env, info)
                        << "checking mailbox for re-creation because of security lock";
                    yield spawn<check_uidvalidity_op>(wrap(env, ctx, uninterruptible));
                    if (ec == code::need_check_timestamp)
                    {
                        yield spawn<check_timestamps_op>(wrap(env, ctx, uninterruptible));
                    }
                    if (ec == code::need_clear_mailbox)
                    {
                        ENV_LOG(env, error) << "clearing mailbox";
                        yield spawn<clear_mailbox_op>(wrap(env, ctx, uninterruptible));
                    }
                    if (ec)
                    {
                        yield break;
                    }
                }

                while (folder = env.cache_mailbox->get_folder_by_status(folder_status::to_init))
                {
                    ENV_LOG(env, info) << "initing system folder as mailish: " << folder->fid << ":"
                                       << folder->path.to_string();

                    yield env.loc_mailbox->init_system_folder(
                        *folder, wrap(env, ctx, uninterruptible));
                    if (!ec)
                    {
                        env.cache_mailbox->mark_inited(folder->path);
                    }
                    else
                    {
                        // TODO process "not empty" error, and force clean mailbox?
                        ENV_LOG(env, error)
                            << "init system folder failed, skipping fid=" << folder->fid
                            << ", reason=\"" << ec.message() << "\"";
                        env.cache_mailbox->set_folder_status_by_path(
                            folder->path, folder_status::sync_error);
                    }
                }

                while (folder = env.cache_mailbox->get_folder_by_status(
                           folder_status::to_create_external))
                {
                    if (!folder->path.empty())
                    {
                        ENV_LOG(env, info)
                            << "creating external folder: " << folder->path.to_string();
                        yield spawn<create_external_folder_op>(wrap(env, ctx), folder->path);
                        if (ec)
                        {
                            ENV_LOG(env, error) << "create external folder failed, skipping path="
                                                << folder->path.to_string() << ", reason=\""
                                                << ec.message() << "\"";
                            env.cache_mailbox->set_folder_status_by_path(
                                folder->path, folder_status::sync_error);
                        }
                    }
                }

                while (folder =
                           env.cache_mailbox->get_folder_by_status(folder_status::to_create_local))
                {
                    ec = {};
                    parent_path = folder->path.get_parent_path();
                    if (parent_path.empty())
                    {
                        parent_fid = "";
                    }
                    else
                    {
                        auto parent_it = env.cache_mailbox->folders().find(parent_path);
                        if (parent_it != env.cache_mailbox->folders().end())
                        {
                            auto& [path, parent] = *parent_it;
                            if (parent.status == folder_status::ok)
                            {
                                parent_fid = parent.fid;
                            }
                            else
                            {
                                ec = code::folder_bad_status;
                            }
                        }
                        else
                        {
                            ec = code::parent_not_found;
                        }
                    }
                    if (!ec)
                    {
                        ENV_LOG(env, info) << "creating local folder: " << folder->path.to_string();
                        yield spawn<create_local_folder_op>(wrap(env, ctx), *folder, parent_fid);
                    }

                    if (ec)
                    {
                        ENV_LOG(env, error)
                            << "create local folder failed, skipping folder="
                            << folder->path.to_string() << ", reason=\"" << ec.message() << "\"";
                        env.cache_mailbox->set_folder_status_by_path(
                            folder->path, folder_status::sync_error);
                    }
                }

                while (folder = env.cache_mailbox->get_folder_by_status(
                           folder_status::to_update_and_clear))
                {
                    ENV_LOG(env, info) << "updating local folder: " << folder->fid;
                    yield spawn<update_and_clear_local_folder_op>(wrap(env, ctx), folder->fid);
                    if (ec)
                    {
                        ENV_LOG(env, error) << "update local folder failed, fid=" << folder->fid
                                            << ", reason=\"" << ec.message() << "\"";
                        env.cache_mailbox->set_folder_status_by_path(
                            folder->path, folder_status::sync_error);
                    }
                }

                while (folder = env.cache_mailbox->get_folder_by_status(folder_status::to_delete))
                {
                    ENV_LOG(env, info) << "deleting local folder: " << folder->fid;
                    yield spawn<delete_local_folder_op>(wrap(env, ctx), *folder);
                    if (ec)
                    {
                        ENV_LOG(env, error) << "delete local folder failed, fid=" << folder->fid
                                            << ", reason=\"" << ec.message() << "\"";
                        env.cache_mailbox->set_folder_status_by_path(
                            folder->path, folder_status::sync_error);
                    }
                }
                ec = code::ok;
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "sync folders op exception: " << e.what();
            ec = code::operation_exception;
        }

        if (ctx.is_complete())
        {
            env(ec);
        }
    }

    bool security_lock{ false };

    mailbox::folder_opt folder;
    mailbox::fid_t parent_fid;
    mailbox::path_t parent_path;
};

}
#include <yplatform/unyield.h>
