#pragma once

#include <common/errors.h>
#include <common/find_deps.h>
#include <mailbox/common.h>
#include <xeno/operations/environment.h>

#include <yplatform/log.h>
#include <yplatform/yield.h>

#include <algorithm>

namespace xeno {

namespace ph = std::placeholders;

struct sync_message_op
{
    using yield_ctx = yplatform::yield_context<sync_message_op>;
    using store_message_response = mailbox::local::store_message_response;

    sync_message_op(
        const mailbox::path_t& path,
        mailbox::imap_id_t ext_imap_id,
        const std::string& parent_operation,
        mailbox::message_opt msg = mailbox::message_opt(),
        mailbox::notification_type notify_type = mailbox::notification_type::normal,
        bool add_mailish_entry_on_errors = true)
        : path(path)
        , ext_imap_id(ext_imap_id)
        , parent_operation(parent_operation)
        , msg(msg)
        , notify_type(notify_type)
        , add_mailish_entry_on_errors(add_mailish_entry_on_errors)
    {
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec = {}, mailbox::string_ptr body = nullptr)
    {
        try
        {
            reenter(ctx)
            {
                fid = env.cache_mailbox->get_fid_by_path(path);
                if (!fid)
                {
                    ec = code::folder_not_found;
                    log_sync_message(env, ec);
                    yield break;
                }

                if (!msg)
                {
                    yield env.ext_mailbox->get_messages_info_by_id(
                        path,
                        mailbox::imap_range(ext_imap_id, ext_imap_id),
                        wrap(env, ctx, uninterruptible));
                    if (ec)
                    {
                        log_sync_message(env, ec);
                        yield break;
                    }
                }
                msg->fid = *fid;

                if (msg->size <= env.sync_settings->max_message_size ||
                    !env.sync_settings->max_message_size)
                {
                    ENV_LOG(env, info)
                        << "downloading body from external mailbox: " << path.to_string()
                        << ", uid: " << ext_imap_id;
                    yield env.ext_mailbox->get_message_body(
                        path, ext_imap_id, wrap(env, ctx, uninterruptible));
                }
                else
                {
                    ec = code::message_too_big;
                }

                if (!ec)
                {
                    if (!body || body->empty())
                    {
                        ec = code::body_is_empty;
                    }

                    if (env.sync_settings->max_message_size &&
                        body->size() > env.sync_settings->max_message_size)
                    {
                        ENV_LOG(env, info)
                            << "downloaded message size is too big: " << body->size();
                        ec = code::message_too_big;
                    }
                }

                if (ec)
                {
                    log_sync_message(env, ec);
                    if (!add_mailish_entry_on_errors)
                    {
                        yield break;
                    }

                    if (ec == errc::imap_not_connected)
                    {
                        need_reconnect = true;
                    }

                    if (!msg->errors_count)
                    {
                        ENV_LOG(env, error) << "downloading error: " << ec.message();
                        env.cache_mailbox->redownload_messages_state()->has_new_failures = true;
                    }

                    env.cache_mailbox->update_message_errors_count(
                        path, ext_imap_id, ++msg->errors_count);
                    ENV_LOG(env, info)
                        << "saving mailish message errors count: " << msg->errors_count;
                    yield env.loc_mailbox->increment_mailish_entry_errors_count(
                        *fid,
                        ext_imap_id,
                        msg->get_errors_count_diff(),
                        wrap(env, ctx, uninterruptible));
                    if (!ec)
                    {
                        msg->saved_errors_count = msg->errors_count;
                        env.cache_mailbox->save_message_errors_count(path, ext_imap_id);
                        ec = code::message_saved_with_error;
                    }
                    ec = need_reconnect ? code::imap_general_error : ec;
                    yield break;
                }

                ENV_LOG(env, info) << "storing message in local mailbox: " << path.to_string();
                env.stat->store_message_attempts++;
                yield env.loc_mailbox->store_message(
                    env.cache_mailbox->account().uid,
                    env.cache_mailbox->account().email,
                    *msg,
                    std::move(*body),
                    notify_type,
                    determine_priority(env),
                    wrap(env, ctx, uninterruptible));
                log_sync_message(env, ec);
                if (ec)
                {
                    env.stat->store_message_errors++;
                    if (!add_mailish_entry_on_errors)
                    {
                        yield break;
                    }

                    if (!msg->errors_count)
                    {
                        ENV_LOG(env, error) << "storing error: " << ec.message();
                        env.cache_mailbox->redownload_messages_state()->has_new_failures = true;
                    }

                    env.cache_mailbox->update_message_errors_count(
                        path, ext_imap_id, ++msg->errors_count);
                    ENV_LOG(env, info)
                        << "saving mailish message errors count: " << msg->errors_count;
                    yield env.loc_mailbox->increment_mailish_entry_errors_count(
                        *fid,
                        ext_imap_id,
                        msg->get_errors_count_diff(),
                        wrap(env, ctx, uninterruptible));
                    if (!ec)
                    {
                        msg->saved_errors_count = msg->errors_count;
                        env.cache_mailbox->save_message_errors_count(path, ext_imap_id);
                        ec = code::message_saved_with_error;
                    }
                    yield break;
                }
                env.cache_mailbox->update_message_if_exists(path, ext_imap_id, msg->mid);
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "sync message op exception: " << e.what();
            ec = code::operation_exception;
        }

        if (ctx.is_complete())
        {
            env(ec, msg ? msg.value() : mb::message());
        }
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec, mailbox::message_vector_ptr messages_res)
    {
        if (!ec)
        {
            if (messages_res->empty())
            {
                ec = code::message_not_found;
            }
            else
            {
                msg = messages_res->front();
            }
        }
        (*this)(ctx, env, ec);
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec, const store_message_response& response)
    {
        if (!ec)
        {
            int_imap_id = response.int_imap_id;
            msg->mid = response.mid;
        }
        (*this)(ctx, env, ec);
    }

    template <typename Env>
    void log_sync_message(Env&& env, error err)
    {
        env.typed_logger->sync_message(
            env.ctx,
            env.cache_mailbox->account().uid,
            path,
            fid,
            ext_imap_id,
            int_imap_id,
            msg,
            parent_operation,
            err);
    }

    template <typename Env>
    std::string determine_priority(Env&& env)
    {
        bool important_phase =
            env.sync_phase == sync_phase::sync_newest || env.sync_phase == sync_phase::user_op;
        if (important_phase && !low_priority_provider(env))
        {
            return "high";
        }
        return "low";
    }

    template <typename Env>
    bool low_priority_provider(Env&& env)
    {
        auto provider = env.ext_mailbox->get_provider_unsafe();
        return env.sync_settings->low_priority_providers.count(provider);
    }

    mailbox::path_t path;
    mailbox::fid_t_opt fid;
    mailbox::imap_id_t ext_imap_id = 0;
    mailbox::imap_id_t int_imap_id = 0;
    std::string parent_operation;
    mailbox::message_opt msg;
    mailbox::notification_type notify_type;
    bool add_mailish_entry_on_errors;
    bool need_reconnect = false;
};

}
#include <yplatform/unyield.h>
