#pragma once

#include "sync_message_op.h"

#include <common/errors.h>
#include <mailbox/common.h>
#include <xeno/operations/environment.h>

#include <yplatform/log.h>
#include <yplatform/yield.h>

#include <algorithm>

namespace xeno {

struct sync_newest_messages_op
{
    using message_status = mailbox::message::status_t;
    using info_type = mailbox::msg_info_type;
    using yield_ctx = yplatform::yield_context<sync_newest_messages_op>;

    sync_newest_messages_op(const mailbox::path_t& path, mailbox::num_t top) : path(path), top(top)
    {
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec = {})
    {
        if (ec)
        {
            saved_ec = ec;
        }
        try
        {
            reenter(ctx)
            {
                state = env.cache_mailbox->sync_newest_state();
                folder = env.cache_mailbox->get_folder_by_path(path);
                if (!folder)
                {
                    ec = code::folder_not_found;
                    yield break;
                }
                bottom = top > env.sync_settings->newest_count ?
                    top - env.sync_settings->newest_count + 1 :
                    1;
                ENV_LOG(env, info) << "getting message list from external mailbox";
                yield env.ext_mailbox->get_messages_info_by_num(
                    path, top, bottom, wrap(env, ctx, uninterruptible));
                if (ec)
                {
                    yield break;
                }

                if (messages->empty())
                {
                    if (is_folder_contain_messages(*folder))
                    {
                        env.cache_mailbox->set_folder_status_by_path(
                            path, mailbox::folder::status_t::to_clear);
                        yield env.loc_mailbox->clear_folder(folder->fid, wrap(env, ctx));
                        if (ec)
                        {
                            ENV_LOG(env, info)
                                << "ignoring error during clearing folder in local mailbox"
                                << ec.message();
                        }
                    }
                    saved_ec = ec = code::ok;
                    yield break;
                }

                ec = update_messages_top_from_external(env.sync_settings->newest_count, messages);
                if (ec)
                {
                    yield break;
                }

                messages = get_messages_by_status(message_status::to_delete);
                if (messages->size())
                {
                    imap_ids = messages_to_imap_ids(messages);
                    ENV_LOG(env, info) << "deleting messages in local mailbox";
                    yield env.loc_mailbox->delete_messages_by_id(
                        folder->fid, *imap_ids, wrap(env, ctx, uninterruptible));
                    if (!ec)
                    {
                        env.cache_mailbox->delete_messages(path, imap_ids);
                    }
                    else
                    {
                        ENV_LOG(env, error)
                            << "ignoring error during deleting messages in local mailbox: "
                            << ec.message();
                        ec = code::ok;
                    }
                }

                messages = get_messages_by_status(message_status::to_load_from_local);
                if (messages->size())
                {
                    imap_ids = messages_to_imap_ids(messages);
                    ENV_LOG(env, info) << "loading top messages from local mailbox";
                    yield env.loc_mailbox->get_messages_info_by_id(
                        folder->fid,
                        *imap_ids,
                        info_type::with_flags,
                        wrap(env, ctx, uninterruptible));
                    if (!ec)
                    {
                        complete_missing_messages_from_local(messages);
                    }
                    else
                    {
                        ENV_LOG(env, error)
                            << "ignoring error during loading top messages from local mailbox: "
                            << ec.message();
                        ec = code::ok;
                    }
                }

                messages = get_messages_by_status(message_status::to_update_flags);
                for (message_it = messages->begin(); message_it != messages->end(); ++message_it)
                {
                    ENV_LOG(env, info)
                        << "updating flags in local mailbox for message: mid=" << message_it->mid;
                    yield env.loc_mailbox->update_flags(
                        folder->fid,
                        message_it->mid,
                        message_it->flags,
                        wrap(env, ctx, uninterruptible));
                    if (ec)
                    {
                        ENV_LOG(env, error)
                            << "ignoring error during updating flags in local mailbox: "
                            << ec.message();
                        ec = code::ok;
                    }
                    else
                    {
                        auto id = env.cache_mailbox->get_imap_id_by_mid(path, message_it->mid);
                        if (id)
                        {
                            update_message_status_by_id(*id, message_status::ok);
                        }
                    }
                }

                messages = get_messages_by_status(mailbox::message::status_t::to_download_body);
                std::sort(messages->rbegin(), messages->rend());

                for (message_it = messages->begin(); message_it != messages->end(); ++message_it)
                {
                    if (message_it->errors_count > env.sync_settings->newest_downloading_retries)
                    {
                        ENV_LOG(env, info) << "skipping message imap_id=" << message_it->id
                                           << ", folder=" << path.to_string()
                                           << ", retries=" << message_it->errors_count;
                        continue;
                    }
                    ENV_LOG(env, info) << "downloading message body";
                    yield spawn<sync_message_op>(
                        wrap(env, ctx),
                        path,
                        message_it->id,
                        "sync_newest_messages",
                        *message_it,
                        (env.cache_mailbox->account().last_sync_ts < message_it->date ?
                             mailbox::notification_type::normal :
                             mailbox::notification_type::disabled));
                    if (ec)
                    {
                        if (ec == errc::imap_not_connected)
                        {
                            // cannot ignore error - reconnect required
                            yield break;
                        }

                        ENV_LOG(env, error)
                            << "ignoring error during downloading body: " << ec.message();
                        ec = code::ok;
                        continue;
                    }
                    ++synced_messages_count;
                }
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "exception during sync_newest_messages_op: " << e.what();
            ec = code::operation_exception;
        }
        if (ctx.is_complete())
        {
            env((ec ? ec : saved_ec), synced_messages_count);
        }
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec, const mailbox::message& /*msg*/)
    {
        (*this)(ctx, std::forward<Env>(env), ec);
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec, mailbox::message_vector_ptr messages)
    {
        if (!ec)
        {
            this->messages = messages;
        }
        (*this)(ctx, std::forward<Env>(env), ec);
    }

    mailbox::imap_id_vector_ptr messages_to_imap_ids(mailbox::message_vector_ptr messages)
    {
        auto res = std::make_shared<mailbox::imap_id_vector>();
        std::transform(
            messages->begin(),
            messages->end(),
            std::back_inserter(*res),
            [](const mailbox::message& msg) { return msg.id; });

        return res;
    }

    bool is_folder_contain_messages(const mailbox::folder& folder)
    {
        auto it = state->folders.find(folder.path);
        return it != state->folders.end() && it->second.messages_top.size();
    }

    void complete_missing_messages_from_local(mailbox::message_vector_ptr messages)
    {
        auto it = state->folders.find(path);
        if (it == state->folders.end())
        {
            return;
        }
        auto& [folder_path, folder_state] = *it;
        for (auto& msg : *messages)
        {
            auto msg_it = folder_state.messages_top.find(msg.id);
            if (msg_it != folder_state.messages_top.end())
            {
                auto& [imap_id, message] = *msg_it;
                if (message.status != mailbox::message::status_t::to_load_from_local) continue;

                message.mid = msg.mid;
                message.errors_count = msg.errors_count;
                message.saved_errors_count = msg.errors_count;
                message.flags = msg.flags;
                message.status = msg.mid ? mailbox::message::status_t::ok :
                                           mailbox::message::status_t::to_download_body;
            }
        }
    }

    error update_messages_top_from_external(
        mailbox::num_t chunk,
        mailbox::message_vector_ptr messages)
    {
        auto it = state->folders.find(path);
        if (it == state->folders.end())
        {
            return code::folder_not_found;
        }

        auto& [folder_path, folder_state] = *it;
        auto min_element = std::min_element(messages->begin(), messages->end());
        auto min_external_message_id = min_element != messages->end() ? min_element->id : 0;

        mailbox::imap_id_vector to_remove;
        for (auto& [id, msg] : folder_state.messages_top)
        {
            auto it = std::find_if(
                messages->begin(), messages->end(), [id = id](const mailbox::message& msg) {
                    return msg.id == id;
                });

            if (it == messages->end())
            {
                if (id > min_external_message_id || messages->size() < chunk)
                {
                    msg.status = mailbox::message::status_t::to_delete;
                }
                else
                {
                    to_remove.push_back(id);
                }
            }
            else
            {
                msg.size = it->size;
                bool can_update_flag = msg.status == mailbox::message::status_t::ok ||
                    msg.status == mailbox::message::status_t::to_update_flags;
                if (can_update_flag && msg.flags != it->flags)
                {
                    msg.flags = it->flags;
                    msg.status = mailbox::message::status_t::to_update_flags;
                }
            }
        }

        auto min_cache_message_id =
            folder_state.messages_top.size() ? folder_state.messages_top.begin()->first : 0;
        for (auto& msg : *messages)
        {
            auto it = folder_state.messages_top.find(msg.id);
            if (it == folder_state.messages_top.end())
            {
                msg.status = msg.id > min_cache_message_id ?
                    mailbox::message::status_t::to_download_body :
                    mailbox::message::status_t::to_load_from_local;
                folder_state.messages_top[msg.id] = msg;
            }
        }

        for (auto& id : to_remove)
        {
            folder_state.messages_top.erase(id);
        }

        return code::ok;
    }

    mailbox::message_vector_ptr get_messages_by_status(mailbox::message::status_t status) const
    {
        auto res = std::make_shared<mailbox::message_vector>();
        auto it = state->folders.find(path);
        if (it != state->folders.end())
        {
            auto& [folder_path, folder_state] = *it;
            for (auto& [imap_id, msg] : folder_state.messages_top)
            {
                if (msg.status == status)
                {
                    res->push_back(msg);
                }
            }
        }
        return res;
    }

    void update_message_status_by_id(mailbox::imap_id_t id, mailbox::message::status_t status)
    {
        auto it = state->folders.find(path);
        if (it != state->folders.end())
        {
            auto& [folder_path, folder_state] = *it;
            auto msg_it = folder_state.messages_top.find(id);
            if (msg_it != folder_state.messages_top.end())
            {
                auto& [imap_id, message] = *msg_it;
                message.status = status;
            }
        }
    }

    mailbox::path_t path;
    mailbox::folder_opt folder;
    mailbox::num_t top;
    mailbox::num_t bottom;

    mailbox::message_vector_ptr messages;
    mailbox::message_vector::iterator message_it;
    mailbox::imap_id_vector_ptr imap_ids;

    size_t synced_messages_count = 0;
    error saved_ec = {};

    mailbox::sync_newest_state_ptr state;
};

}
#include <yplatform/unyield.h>
