#pragma once

#include "load_folder_top_op.h"
#include "sync_newest_messages_op.h"
#include "turbo_sync_op.h"

#include <common/errors.h>
#include <mailbox/common.h>
#include <mailbox/data_types/sync_newest_state.h>
#include <xeno/operations/environment.h>

#include <yplatform/log.h>
#include <yplatform/yield.h>

#include <algorithm>
#include <memory>

namespace xeno {

struct sync_newest_op
{
    using message_status = mailbox::message::status_t;
    using yield_ctx = yplatform::yield_context<sync_newest_op>;

    sync_newest_op(const mailbox::path_vector& paths_to_sync) : paths_to_sync(paths_to_sync)
    {
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec = {})
    {
        if (ec)
        {
            sync_without_error = false;
        }

        try
        {
            reenter(ctx)
            {
                {
                    auto inbox_path_opt =
                        env.cache_mailbox->get_path_by_folder_type(mailbox::folder::type_t::inbox);
                    if (inbox_path_opt)
                    {
                        inbox_path = *inbox_path_opt;
                    }
                }
                state = env.cache_mailbox->sync_newest_state();
                reorder_paths();
                for (path_it = paths_to_sync.begin(); path_it < paths_to_sync.end(); ++path_it)
                {
                    folder = env.cache_mailbox->get_folder(*path_it);
                    if (state->folders.find(*path_it) == state->folders.end())
                    {
                        yield spawn<load_folder_top_op>(wrap(env, ctx), folder);
                        if (ec)
                        {
                            ENV_LOG(env, info)
                                << "ignoring error during load folder top error=" << ec.message()
                                << " path=" << path_it->to_string();
                            ec = code::ok;
                            continue;
                        }
                    }

                    if (folder.status != mailbox::folder::status_t::ok) continue;
                    if (*path_it != inbox_path)
                    {
                        // change state->current_folder_path in case that it is not inbox
                        state->current_folder_path = *path_it;
                    }
                    ENV_LOG(env, info)
                        << "processing folder, name=\"" << path_it->to_string() << "\"";
                    prev_uidnext = folder.uidnext;
                    yield env.ext_mailbox->get_folder_info(*path_it, wrap(env, ctx));

                    if (!ec)
                    {
                        ec = env.cache_mailbox->update_folder_info_from_external(*ext_folder);
                    }

                    if (ec)
                    {
                        if (ec == errc::imap_not_connected)
                        {
                            // cannot ignore error - reconnect required
                            yield break;
                        }
                        else
                        {
                            ENV_LOG(env, info)
                                << "ignoring error during sync newest error=" << ec.message();
                            ec = code::ok;
                            continue;
                        }
                    }

                    // get folder with actualized info
                    folder = env.cache_mailbox->get_folder(*path_it);

                    if (too_many_new_messages(env.sync_settings))
                    {
                        state->turbo_sync.enable(folder, env.sync_settings->turbo_sync);
                    }

                    if (sync_newest_messages_required(env.sync_settings))
                    {
                        yield spawn<sync_newest_messages_op>(
                            wrap(env, ctx), *path_it, ext_folder->count);
                        if (ec)
                        {
                            if (ec == errc::imap_not_connected)
                            {
                                // cannot ignore error - reconnect required
                                yield break;
                            }

                            ENV_LOG(env, error)
                                << "ignoring error during sync newest messages: " << ec.message();
                            ec = code::ok;
                        }
                    }
                    unlock_api_read(env);

                    if (turbo_sync_required(env.sync_settings))
                    {
                        state->turbo_sync.folders[folder.path].status =
                            mailbox::turbo_sync_status::running;
                        yield spawn<turbo_sync_op>(wrap(env, ctx), folder);
                        if (!ec)
                        {
                            ENV_LOG(env, info) << "turbo_sync finished";
                            state->turbo_sync.folders[folder.path].status =
                                mailbox::turbo_sync_status::finished;
                        }
                        else
                        {
                            if (ec == errc::imap_not_connected)
                            {
                                // cannot ignore error - reconnect required
                                yield break;
                            }

                            ENV_LOG(env, error)
                                << "ignoring error during turbo_sync: " << ec.message();
                            ec = code::ok;
                        }
                    }
                    state->folders[folder.path].last_sync_ts = time_traits::clock::now();
                }

                if (synced_messages_count > 0 && sync_without_error)
                {
                    yield env.loc_mailbox->update_last_sync_ts(
                        std::time(nullptr), wrap(env, ctx, uninterruptible));
                }
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "exception during sync_newest: " << e.what();
            ec = code::operation_exception;
        }
        if (ctx.is_complete())
        {
            env(ec);
        }
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec, mailbox::folder_ptr folder)
    {
        ext_folder = folder;
        (*this)(ctx, std::forward<Env>(env), ec);
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec, size_t messages_count)
    {
        synced_messages_count = messages_count;
        (*this)(ctx, std::forward<Env>(env), ec);
    }

    template <typename Env>
    void unlock_api_read(const Env& env)
    {
        try
        {
            auto& folder = env.cache_mailbox->get_folder(*path_it);
            folder.api_read_lock = false;
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "exception during sync_newest: " << e.what();
        }
    }

    void reorder_paths()
    {
        mailbox::path_t inbox;
        auto path_to_start = state->current_folder_path;
        auto sync_pos = paths_to_sync.end();
        for (auto path = paths_to_sync.begin(); path != paths_to_sync.end(); ++path)
        {
            if (sync_pos == paths_to_sync.end() && *path >= path_to_start)
            {
                sync_pos = path;
            }
            if (*path == inbox_path)
            {
                inbox = *path;
                continue;
            }
        }

        mailbox::path_vector ordered_paths_to_sync;
        if (!inbox.empty())
        {
            ordered_paths_to_sync.emplace_back(inbox);
        }
        for (auto elem = sync_pos; elem != paths_to_sync.end(); ++elem)
        {
            if (*elem != inbox_path)
            {
                ordered_paths_to_sync.emplace_back(*elem);
            }
        }
        for (auto elem = paths_to_sync.begin(); elem != sync_pos; ++elem)
        {
            if (*elem != inbox_path)
            {
                ordered_paths_to_sync.emplace_back(*elem);
            }
        }
        paths_to_sync = std::move(ordered_paths_to_sync);
    }

    bool too_many_new_messages(synchronization_settings_ptr settings)
    {
        return ext_folder->uidnext - prev_uidnext > settings->newest_count &&
            ext_folder->count > settings->newest_count;
    }

    bool sync_newest_messages_required(synchronization_settings_ptr settings)
    {
        bool has_messages = ext_folder->count > 0;
        bool turbo_sync_running = state->turbo_sync.enabled(folder, settings->turbo_sync) &&
            state->turbo_sync.folders[folder.path].status == mailbox::turbo_sync_status::running;

        return has_messages && !turbo_sync_running;
    }

    bool turbo_sync_required(synchronization_settings_ptr settings)
    {
        return state->turbo_sync.enabled(folder, settings->turbo_sync);
    }

    mailbox::path_vector paths_to_sync;
    mailbox::path_vector::iterator path_it;
    mailbox::path_t inbox_path{};
    mailbox::sync_newest_state_ptr state;

    mailbox::folder_ptr ext_folder;
    mailbox::folder folder;
    mailbox::imap_id_t prev_uidnext = 0;
    size_t synced_messages_count = 0;
    bool sync_without_error = true;
};

}
#include <yplatform/unyield.h>
