#pragma once

#include "sync_oldest_messages_op.h"

#include <common/errors.h>
#include <xeno/operations/environment.h>

#include <yplatform/log.h>
#include <yplatform/yield.h>

#include <memory>

namespace xeno {

struct sync_oldest_flags_and_deleted_op
{
    using yield_ctx = yplatform::yield_context<sync_oldest_flags_and_deleted_op>;

    sync_oldest_flags_and_deleted_op(mailbox::imap_id_t chunk) : chunk(chunk)
    {
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec = {})
    {
        try
        {
            reenter(ctx)
            {
                state = env.cache_mailbox->sync_oldest_flags_and_deleted_state();
                folders = env.cache_mailbox->folders_copy();
                if (folders->empty())
                {
                    yield break;
                }
                actualize_folder_list();
                folder = find_position_to_start_sync(state->current_folder_path);
                state->current_folder_path = folder->path;

                while (processed_folders_count < folders->size())
                {
                    if (folder->count && folder->status == mailbox::folder::status_t::ok)
                    {
                        ENV_LOG(env, info) << "syncing oldest messages flags and deleted, path: "
                                           << folder->path.to_string() << ", count=" << chunk;
                        yield spawn<sync_oldest_messages_op>(wrap(env, ctx), folder->path, chunk);
                        if (ec)
                        {
                            if (ec == errc::imap_not_connected)
                            {
                                // cannot ignore error - reconnect required
                                yield break;
                            }
                            ENV_LOG(env, error) << "ignoring error during syncing oldest messages "
                                                   "flags and deleted: "
                                                << ec.message();
                            ec = code::ok;
                        }
                    }
                    ++processed_folders_count;
                    if (++folder == folders->end())
                    {
                        folder = folders->begin();
                    }
                    state->current_folder_path = folder->path;
                }
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "sync oldest flags and deleted op exception: " << e.what();
            ec = code::operation_exception;
        }

        if (ctx.is_complete())
        {
            env(ec);
        }
    }

    void actualize_folder_list()
    {
        std::map<mailbox::path_t, mailbox::imap_id_t> sync_positions;

        for (auto& folder : *folders)
        {
            // for new folders we put default value
            sync_positions[folder.path] = state->sync_positions[folder.path];
        }
        state->sync_positions = std::move(sync_positions);
    }

    mailbox::folder_vector::iterator find_position_to_start_sync(const mailbox::path_t& path)
    {
        auto folder_it =
            std::find_if(folders->begin(), folders->end(), [&path](const mailbox::folder& folder) {
                return path >= folder.path;
            });
        if (folder_it == folders->end())
        {
            folder_it = folders->begin();
        }
        return folder_it;
    }

    size_t processed_folders_count = 0;
    mailbox::imap_id_t chunk;

    mailbox::folder_vector_ptr folders;
    mailbox::folder_vector::iterator folder;

    mailbox::sync_oldest_flags_and_deleted_state_ptr state;
};

}
#include <yplatform/unyield.h>
