#pragma once

#include <common/errors.h>
#include <mailbox/common.h>
#include <xeno/operations/environment.h>

#include <yplatform/log.h>
#include <yplatform/yield.h>

namespace xeno {

struct sync_oldest_messages_op
{
    using yield_ctx = yplatform::yield_context<sync_oldest_messages_op>;

    sync_oldest_messages_op(const mailbox::path_t& path, size_t count) : path(path), count(count)
    {
    }

    template <typename Env>
    void operator()(
        yield_ctx ctx,
        Env&& env,
        error ec = {},
        mailbox::message_vector_ptr messages = nullptr)
    {
        try
        {
            reenter(ctx)
            {
                fid = env.cache_mailbox->get_fid_by_path(path);
                if (!fid)
                {
                    return env(code::folder_not_found);
                }

                state = env.cache_mailbox->sync_oldest_flags_and_deleted_state();
                highest_imap_id = state->sync_positions[path];
                correct_highest_imap_id(env, path);

                ENV_LOG(env, info) << "getting messages list from local mailbox, fid=" << *fid
                                   << ", highest_imap_id=" << highest_imap_id;
                yield env.loc_mailbox->get_messages_info_chunk_by_id(
                    *fid, highest_imap_id, count, wrap(env, ctx, uninterruptible));
                if (ec)
                {
                    yield break;
                }

                if (messages->empty())
                {
                    state->sync_positions[path] = 0;
                    ec = code::ok;
                    yield break;
                }
                loc_messages = messages;

                {
                    auto [min, max] =
                        std::minmax_element(loc_messages->begin(), loc_messages->end());
                    imap_range = mailbox::imap_range(max->id, min->id);
                }

                ENV_LOG(env, info)
                    << "getting messages list from external mailbox, path=" << path.to_string()
                    << ", imap_range=" << imap_range.top() << ":" << imap_range.bottom();
                yield env.ext_mailbox->get_messages_info_by_id(
                    path, imap_range, wrap(env, ctx, uninterruptible));
                if (ec)
                {
                    yield break;
                }
                ext_messages = messages;

                for (imsg_loc = loc_messages->begin(); imsg_loc != loc_messages->end(); ++imsg_loc)
                {
                    if (!imsg_loc->mid) continue;
                    imsg_ext = std::find_if(
                        ext_messages->begin(),
                        ext_messages->end(),
                        [this](const mailbox::message& ext_msg) {
                            return imsg_loc->id == ext_msg.id;
                        });

                    if (imsg_ext == ext_messages->end())
                    {
                        del_mids.push_back(imsg_loc->mid);
                    }
                    else if (imsg_ext->flags != imsg_loc->flags)
                    {
                        ENV_LOG(env, info)
                            << "updating message flags in local mailbox, mid: " << imsg_loc->mid;
                        yield env.loc_mailbox->update_flags(
                            *fid, imsg_loc->mid, imsg_ext->flags, wrap(env, ctx, uninterruptible));
                        if (ec)
                        {
                            yield break;
                        }
                    }
                }

                if (!del_mids.empty())
                {
                    ENV_LOG(env, info)
                        << "deleting messages from local mailbox, count: " << del_mids.size();
                    yield env.loc_mailbox->delete_messages_by_mid(
                        del_mids, wrap(env, ctx, uninterruptible));
                    if (ec)
                    {
                        yield break;
                    }
                }
                state->sync_positions[path] = get_imap_id_for_next_iteration();
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "sync oldest messages op exception: " << e.what();
            ec = code::operation_exception;
        }

        if (ctx.is_complete())
        {
            env(ec);
        }
    }

    mailbox::imap_id_t get_imap_id_for_next_iteration()
    {
        return imap_range.bottom() - 1;
    }

    template <typename Env>
    void correct_highest_imap_id(const Env& env, const mailbox::path_t& path)
    {
        if (!highest_imap_id)
        {
            auto min_imap_id = env.cache_mailbox->get_min_imap_id_from_top(path);
            highest_imap_id = (min_imap_id ? *min_imap_id : 0);
        }
    }

    mailbox::path_t path;
    mailbox::fid_t_opt fid;
    mailbox::imap_id_t highest_imap_id;
    size_t count;

    mailbox::imap_range imap_range;
    mailbox::message_vector_ptr ext_messages;
    mailbox::message_vector::iterator imsg_ext;

    mailbox::message_vector_ptr loc_messages;
    mailbox::message_vector::iterator imsg_loc;

    mailbox::mid_vector del_mids;

    mailbox::sync_oldest_flags_and_deleted_state_ptr state;
};

}
#include <yplatform/unyield.h>
