#pragma once

#include "sync_message_op.h"
#include <xeno/operations/environment.h>

#include <yplatform/yield.h>

namespace xeno {

inline mailbox::message_vector_ptr remove_already_downloaded_messages(
    mailbox::message_vector_ptr external,
    mailbox::message_vector_ptr local)
{
    auto res = std::make_shared<mailbox::message_vector>(*external);
    for (auto& loc_msg : *local)
    {
        auto msg_it =
            std::find_if(res->begin(), res->end(), [&loc_msg](const mailbox::message& msg) {
                return loc_msg.id == msg.id;
            });
        if (msg_it != res->end())
        {
            res->erase(msg_it);
        }
    }
    return res;
}

struct sync_oldest_messages_top_op
{
    using msg_info_type = mailbox::msg_info_type;
    using message_status = mailbox::message::status_t;
    using yield_ctx = yplatform::yield_context<sync_oldest_messages_top_op>;

    sync_oldest_messages_top_op(const mailbox::folder& folder) : folder(folder)
    {
    }

    template <typename Env>
    void operator()(
        yield_ctx ctx,
        Env&& env,
        error ec = {},
        mailbox::message_vector_ptr messages = {})
    {
        try
        {
            reenter(ctx)
            {
                if (!folder.downloaded_range.top()) yield break;

                range_to_sync = calc_range_to_sync(
                    env.cache_mailbox->get_min_imap_id_from_top(folder.path),
                    env.sync_settings->oldest_count);
                if (!range_to_sync) yield break;

                ENV_LOG(env, info)
                    << "getting message list from external mailbox: " << folder.path.to_string();
                yield env.ext_mailbox->get_messages_info_by_id(
                    folder.path, *range_to_sync, wrap(env, ctx));
                if (ec)
                {
                    if (ec != errc::imap_not_connected)
                    {
                        ENV_LOG(env, error) << "ignoring error during sync_oldest_messages_top_op: "
                                            << ec.message();
                        ec = {};
                    }
                    yield break;
                }

                if (messages->size())
                {
                    ext_messages = messages;
                    external_ids = mailbox::message_helpers::to_imap_id_vector(ext_messages);
                    ENV_LOG(env, info) << "getting message list from local mailbox";
                    yield env.loc_mailbox->get_messages_info_by_id(
                        folder.fid, *external_ids, msg_info_type::without_flags, wrap(env, ctx));
                    if (ec) yield break;
                    loc_messages = messages;
                    ext_messages = remove_already_downloaded_messages(ext_messages, loc_messages);

                    for (imsg = ext_messages->begin(); imsg != ext_messages->end();
                         ++imsg, ++downloaded_count)
                    {
                        yield spawn<sync_message_op>(
                            wrap(env, ctx),
                            folder.path,
                            imsg->id,
                            "sync_oldest_messages_top",
                            *imsg,
                            mailbox::notification_type::disabled);
                        if (ec == code::message_saved_with_error)
                        {
                            ec = code::ok;
                        }
                        if (ec) yield break;
                    }
                }
                env.cache_mailbox->update_downloaded_range(
                    folder.path, { range_to_sync->top(), folder.downloaded_range.top() });
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "exception during sync_folder_top: " << e.what();
            ec = code::operation_exception;
        }
        if (ctx.is_complete())
        {
            env(ec, downloaded_count);
        }
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec, const mailbox::message& /*msg*/)
    {
        (*this)(ctx, env, ec);
    }

    mailbox::imap_range_opt calc_range_to_sync(
        mailbox::imap_id_opt min_imap_id_from_top,
        mailbox::num_t messages_limit)
    {
        if (min_imap_id_from_top && *min_imap_id_from_top > folder.downloaded_range.top())
        {
            auto top =
                std::min(*min_imap_id_from_top - 1, folder.downloaded_range.top() + messages_limit);
            return mailbox::imap_range{ top, folder.downloaded_range.top() + 1 };
        }
        return {};
    }

    mailbox::folder folder;
    mailbox::imap_range_opt range_to_sync;
    mailbox::message_vector_ptr ext_messages;
    mailbox::message_vector_ptr loc_messages;
    mailbox::message_vector::iterator imsg;
    mailbox::imap_id_vector_ptr external_ids;
    std::size_t downloaded_count = 0;
};

}
#include <yplatform/unyield.h>
