#pragma once

#include "sync_oldest_messages_top_op.h"
#include "update_oldest_messages_cache_op.h"
#include "sync_message_op.h"

#include <common/errors.h>
#include <mailbox/common.h>
#include <xeno/operations/environment.h>

#include <yplatform/log.h>
#include <yplatform/yield.h>

#include <algorithm>
#include <memory>

namespace xeno {

struct sync_oldest_op
{
    using yield_ctx = yplatform::yield_context<sync_oldest_op>;

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec = {}, std::size_t downloaded_count = 0)
    {
        try
        {
            reenter(ctx)
            {
                folders = env.cache_mailbox->folders_copy();
                std::sort(
                    folders->begin(),
                    folders->end(),
                    [](const mailbox::folder& lhs, const mailbox::folder& rhs) {
                        return lhs.importance < rhs.importance;
                    });

                for (ifolder = folders->begin(); ifolder != folders->end(); ++ifolder)
                {
                    if (ifolder->status != mailbox::folder::status_t::ok) continue;

                    ENV_LOG(env, info)
                        << "processing folder, name=\"" << ifolder->path.to_string() << "\"";
                    yield spawn<sync_oldest_messages_top_op>(wrap(env, ctx), *ifolder);
                    if (ec)
                    {
                        ENV_LOG(env, error) << "skipping folder: " << ifolder->path.to_string()
                                            << " - error:" << ec.message();
                        ec = {};
                        continue;
                    }

                    first_downloaded_id = 0;
                    actual_range = env.cache_mailbox->get_downloaded_range(ifolder->path);
                    messages_limit = env.sync_settings->oldest_count - downloaded_count;

                    messages_cache = env.cache_mailbox->get_oldest_messages_for_sync(ifolder->path);
                    if (!messages_cache || messages_cache->empty())
                    {
                        yield spawn<update_oldest_messages_cache_op>(wrap(env, ctx), *ifolder);
                        if (ec)
                        {
                            ENV_LOG(env, error) << "skipping folder: " << ifolder->path.to_string()
                                                << " - error " << ec.message();
                            ec = {};
                            continue;
                        }
                        messages_cache =
                            env.cache_mailbox->get_oldest_messages_for_sync(ifolder->path);
                    }

                    if (messages_cache->empty())
                    {
                        update_downloaded_ids(1);
                    }

                    while (messages_cache->size() && messages_limit--)
                    {
                        ENV_LOG(env, info) << "downloading message body";
                        yield spawn<sync_message_op>(
                            wrap(env, ctx),
                            ifolder->path,
                            messages_cache->back().id,
                            "sync_oldest",
                            messages_cache->back(),
                            mailbox::notification_type::disabled);
                        if (ec == code::message_saved_with_error)
                        {
                            ec = code::ok;
                        }
                        if (ec) yield break;
                        update_downloaded_ids(messages_cache->back().id);
                        messages_cache->pop_back();
                    }

                    env.cache_mailbox->update_downloaded_range(
                        ifolder->path,
                        { actual_range ? actual_range->bottom() : first_downloaded_id,
                          last_downloaded_id });
                    actual_range = env.cache_mailbox->get_downloaded_range(ifolder->path);
                    if (actual_range && ifolder->downloaded_range != *actual_range)
                    {
                        ENV_LOG(env, info) << "updating downloaded range: (" << actual_range->top()
                                           << "," << actual_range->bottom() << ")";
                        env.stat->downloaded_range_updated = true;
                        yield env.loc_mailbox->update_downloaded_range(
                            ifolder->fid, *actual_range, wrap(env, ctx));
                        if (ec) yield break;
                    }
                }
            }
        }
        catch (const std::exception& e)
        {
            ENV_LOG(env, error) << "exception during sync_oldest: " << e.what();
            ec = code::operation_exception;
        }
        if (ctx.is_complete())
        {
            env(ec);
        }
    }

    template <typename Env>
    void operator()(yield_ctx ctx, Env&& env, error ec, const mailbox::message& /*msg*/)
    {
        (*this)(ctx, env, ec);
    }

    void update_downloaded_ids(mailbox::imap_id_t id)
    {
        last_downloaded_id = id;
        if (!first_downloaded_id)
        {
            first_downloaded_id = last_downloaded_id;
        }
    }

    mailbox::imap_id_t messages_limit;

    mailbox::folder_vector_ptr folders;
    mailbox::folder_vector::iterator ifolder;

    mailbox::message_vector_ptr messages_cache;
    mailbox::imap_id_t first_downloaded_id = 0;
    mailbox::imap_id_t last_downloaded_id = 0;

    mailbox::imap_range_opt actual_range;
};

}
#include <yplatform/unyield.h>
